package maito.integration;

import maito.LogListener;
import maito.datacollecting.DataSourceDescription;
import maito.datacollecting.Record;
import maito.util.Tools;
import maito.util.DbTools;

import java.io.File;
import java.sql.*;
import java.util.Properties;
import java.util.Vector;

/**
 * @author Tuomas Tanner
 * @author Oskari Saarekas
 * 
 * This software is released under the GNU GPL license
 */
public class IntegratorImpl implements Integrator {

    private IntegratorHelper iHelper;
    private Properties dbConfig;
    private int integrationTasks = 0;
    
    /**
     * Loads the database config into memory.
     * 
     */
    public IntegratorImpl() {

        // load config file
        dbConfig = Tools.loadProperties(Tools.PATH_DBCONFIG);
        if (dbConfig == null) {
            System.out.println("ERROR: Could not load dbconfig.properties");
        }
        // load document fieldmap file
        Properties docFieldmap = Tools.loadProperties("config" + File.separator + "fieldmap_document.properties");
        if (docFieldmap == null) {
            System.out.println("ERROR: Could not load fieldmap_document.properties");
        }
        // load actor & channel fieldmap file
        Properties actorFieldmap = Tools.loadProperties("config" + File.separator + "fieldmap_actorchannel.properties");
        if (actorFieldmap == null) {
            System.out.println("ERROR: Could not load fieldmap_document.properties");
        }
        // load quicknameformat fieldmap file
        Properties quickNameFieldmap = Tools.loadProperties("config" + File.separator + "fieldmap_qnformat.properties");
        if (quickNameFieldmap == null) {
            System.out.println("ERROR: Could not load fieldmap_qnformat.properties");
        }
        
        this.iHelper = new IntegratorHelper(docFieldmap, actorFieldmap, quickNameFieldmap);
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.integration.Integrator#addGraph(java.lang.String)
     */
    public void addGraph(String graphName) throws IllegalArgumentException {
        if (graphName == null || graphName.length() < 1) {
            throw new IllegalArgumentException("graphName was null or empty");
        }
        String resDbName = dbConfig.getProperty("dbname_resources") + "_"
                + graphName;

        Connection con = null;
        Statement stm = null;
        ResultSet rs = null;
        try {
            con = DbTools.createDbConnection(dbConfig);
            stm = con.createStatement();
            rs = stm.executeQuery("SHOW DATABASES LIKE '" + resDbName + "'");
            if (!rs.next()) { //database not there - create it

                String sqlScript = Tools.readFile(Tools.PATH_RESOURCEGRAPH_SQL);
                if (sqlScript == null) {
                    System.out.println("ERROR: Error loading sql script file: resourcegraph.sql");
                    return;
                }
                stm.executeUpdate("CREATE DATABASE IF NOT EXISTS " + resDbName
                        + " CHARACTER SET utf8");
                stm.executeUpdate("USE " + resDbName);
                debug("Creating database from script");
                // execute each sql command from script
                int previous = 0;
                for (int i = sqlScript.indexOf(";", previous); i != -1; 
                         i = sqlScript.indexOf(";", previous)) {

                    String query = sqlScript.substring(previous, i);
                    debug("SQL: " + query);
                    stm.executeUpdate(query);
                    previous = i + 1;
                }
            } //else database already created

        } catch (SQLException e) {
            System.out.println("ERROR: " + e.getMessage());
        } finally {
            try {
                if (rs != null)
                    rs.close();
            } catch (SQLException e) {}
            try {
                if (stm != null)
                    stm.close();
            } catch (SQLException e) {}
            try {
                if (con != null)
                    con.close();
            } catch (SQLException e) {}
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.integration.Integrator#getGraphs()
     */
    public java.util.List getGraphs() {
        Vector graphs = new Vector();
        
        Connection con = null;
        Statement stm = null;
        ResultSet rs = null;
        
        try {
            con = DbTools.createDbConnection(dbConfig);
            stm = con.createStatement();
            rs = stm.executeQuery("SHOW DATABASES LIKE '" +
                    dbConfig.getProperty("dbname_resources") + "%'");
            while (rs.next()) {
                String graph = rs.getString(1);
                int prefixLen = dbConfig.getProperty("dbname_resources").length() + 1;
                graphs.add(graph.substring(prefixLen));
            }
        } catch (SQLException e) {
            System.out.println("ERROR: " + e.getMessage());
        } finally {
            try {
                if (rs != null)
                    rs.close();
            } catch (SQLException e) {}
            try {
                if (stm != null)
                    stm.close();
            } catch (SQLException e) {}
            try {
                if (con != null)
                    con.close();
            } catch (SQLException e) {}
        }
        
        return graphs;
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see maito.integration.Integrator#getSources(java.lang.String)
     */
    public java.util.List getSources(String graphName)
            throws IllegalArgumentException {

        if (graphName == null || graphName.length() < 1) {
            throw new IllegalArgumentException("graphName was null or empty");
        }
        String resDbName = dbConfig.getProperty("dbname_resources") + "_"
                + graphName;

        Vector sources = new Vector();
        
        Connection con = null;
        Statement stm = null;
        ResultSet rs = null;
        
        try {
            con = DbTools.createDbConnection(dbConfig);
            stm = con.createStatement();
            stm.execute("USE " + resDbName);
            rs = stm.executeQuery("SELECT sourceID, integrated FROM DataSources");
            while (rs.next()) {
                String[] tempArr = {rs.getString(1), rs.getString(2)};
                sources.add(tempArr);
            }
        } catch (SQLException e) {
            System.out.println("ERROR: " + e.getMessage());
        } finally {
            try {
                if (rs != null)
                    rs.close();
            } catch (SQLException e) {}
            try {
                if (stm != null)
                    stm.close();
            } catch (SQLException e) {}
            try {
                if (con != null)
                    con.close();
            } catch (SQLException e) {}
        }
        
        return sources;
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.integration.Integrator#integrate(maito.datacollecting.DataSourceDescription,
     *      java.lang.String)
     */
    public void integrate(DataSourceDescription[] sources, String graph)
            throws IntegrationException {
        //run integration
        this.taskStarted();
        IntegrationThread it = new IntegrationThread(this, sources, graph);
        it.start();
        System.out.println("Integration started");
    }
    
    /**
     *  The actual method that does the integration
     * @param rr RecordReader
     * @param con Connection to resource graph
     * @param sources the array of data sources that we are going to integrate
     */
    private void doIntegration(DataSourceDescription[] sources, String graph) {
        String rawDbName = dbConfig.getProperty("dbname_rawdata");
        String resDbName = dbConfig.getProperty("dbname_resources") + "_" + graph;
        String query = null;
        
        Connection rawCon = null;
        Connection resCon = null;
        Statement rawStm = null;
        Statement resStm = null;
        ResultSet resRs = null;
        try {
            //setup RecordReader
            System.out.println("raw db: " + rawDbName);
            rawCon = DbTools.createDbConnection(dbConfig, rawDbName);
            rawStm = rawCon.createStatement();
            RecordReader rr = new RecordReader(rawCon);
            
            //setup connection to Database
            resCon = DbTools.createDbConnection(dbConfig, resDbName);
            resStm = resCon.createStatement();
            
            /*
             * Go through each datasource
             */
            for (int i = 0; i < sources.length; ++i ) {
                String sourceID = sources[i].getId();
                //add current datasource entry to graph if not there yet
                query = "SELECT * FROM DataSources where sourceID=" +
                    DbTools.SQLstr(sourceID, DataSourceDescription.IDLENGTH);
                debug(query);
                resRs = resStm.executeQuery(query);
                if (!resRs.next()) {
                    query = "INSERT INTO DataSources SET sourceID=" +
                        DbTools.SQLstr(sourceID, DataSourceDescription.IDLENGTH) +
                        ", integrated=CURDATE()";
                    debug(query);
                    resStm.executeUpdate(query);
                }
                
                /*
                 * Go through all the records in this datasource
                 */
                rr.getRecords(sources[i]);
                Record curRecord = null;
                while ((curRecord = rr.nextRecord()) != null) { 
                    String curID = curRecord.getID();
                    
                    /*
                     * If document record has already been integrated
                     * to this graph, skip it. Otherwise mark it only
                     * if it is of quickname format. Document type
                     * records will be marked later on
                     */
                    String integratedTo = curRecord.getIntegratedTo();
                    if (integratedTo == null) {
                        integratedTo = "";
                    }
                    if (integratedTo.indexOf(graph) == -1) {
                        if (sources[i].getDataFormat().equals("quick_format_name")) {
                            query = "UPDATE DataRecord SET integratedTo='" + 
                                integratedTo + graph +  ",' WHERE id=" +
                                DbTools.SQLstr(curID, Record.IDLENGTH) + 
                                " AND source=" + DbTools.SQLstr(sourceID,
                                DataSourceDescription.IDLENGTH);
                            debug(query);
                            rawStm.executeUpdate(query);
                        }
                    }
                    else {
                        debug("*** Record already integrated, continuing");
                        continue;
                    }
                    
                    if (sources[i].getDataFormat().equals("quick_format_name")) {
                        debug("Starting name format integration");
                        try {
                            iHelper.integrateNameQuickFormat(resCon, sourceID, curRecord);
                        }
                        catch (IntegrationException e) {
                            System.out.println("An exception occured when trying to " +
                                "integrate a namequickformat record:");
                            System.out.println(e.getMessage());
                            System.out.println("Continuing anyway...");
                        }
                    }
                    else { //document format
                        debug("*** Starting document integration");
                        try {
                            iHelper.integrateDocument(resCon, sourceID, curRecord);
                        }
                        catch (IntegrationException e) {
                            System.out.println("An exception occured when trying to " +
                                "integrate a document record:");
                            System.out.println(e.getMessage());
                            System.out.println("Continuing anyway...");
                        }
                    }
                } //end while (foreach record)
                
                /*
                 * If source format is not name quickformat,
                 * go through records again to resolve references
                 */
                if (!sources[i].getDataFormat().equals("quick_format_name")) {
                    rr.getRecords(sources[i]);
                    while ((curRecord = rr.nextRecord()) != null) { 
                        /*
                         * If document record has already been integrated
                         * to this graph, skip it. Otherwise mark it, and
                         * resolve references to other documents
                         */
                        String integratedTo = curRecord.getIntegratedTo();
                        if (integratedTo == null) {
                            integratedTo = "";
                        }
                        if (integratedTo.indexOf(graph) != -1) {
                            debug("*** DocRecord already integrated (skip)");
                            continue;
                        }

                        query = "UPDATE DataRecord SET integratedTo='" + 
                            integratedTo + graph +  ",' WHERE id=" +
                            DbTools.SQLstr(curRecord.getID(), Record.IDLENGTH)+
                            " AND source=" + DbTools.SQLstr(sourceID,
                            DataSourceDescription.IDLENGTH);
                        debug(query);
                        rawStm.executeUpdate(query);
                
                        debug("*** Starting to resolve document references");
                        try {
                            iHelper.resolveDocumentReferences(resCon, curRecord, sourceID);
                        }
                        catch (IntegrationException e) {
                            System.out.println("An exception occured when trying to " +
                                "resolve document references:");
                            System.out.println(e.getMessage());
                                    System.out.println("Continuing anyway...");
                        }
                    }
                }
            }
            rr.close();
        } catch (SQLException e) {
            System.out.println("doIntSQLERROR: " + e.getMessage());
            e.printStackTrace();
        } finally {
            try {
                if (rawStm != null)
                    rawStm.close();
            } catch (SQLException e) {}
            try {
                if (rawCon != null)
                    rawCon.close();
            } catch (SQLException e) {}
            try {
                if (resRs != null)
                    resRs.close();
            } catch (SQLException e) {}
            try {
                if (resStm != null)
                    resStm.close();
            } catch (SQLException e) {}
            try {
                if (resCon != null)
                    resCon.close();
            } catch (SQLException e) {}
        }
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see maito.integration.Integrator#setLogListener(java.lang.String,
     *      maito.LogListener)
     */
    public String setLogListener(String graph, LogListener listener) {
        // TODO Auto-generated method stub
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.DataProcessor#workInProgress()
     */
    public boolean workInProgress() {
        return (this.integrationTasks > 0);
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.DataProcessor#getCurrentTasks()
     */
    public String[] getCurrentTasks() {
        // TODO Auto-generated method stub
        return null;
    }
    
    protected synchronized void taskStarted() {
        ++this.integrationTasks;
    }

    protected synchronized void taskEnded() {
        --this.integrationTasks;
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.DataProcessor#getErrors()
     */
    public String[] getErrors() {
        // TODO Auto-generated method stub
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see maito.DataProcessor#setLogListener(maito.LogListener)
     */
    public void setLogListener(LogListener listener) {
        // TODO Auto-generated method stub

    }

    class IntegrationThread extends Thread {

        private IntegratorImpl parent;
        private DataSourceDescription[] sources;
        private String graph;

        public IntegrationThread(IntegratorImpl parent, DataSourceDescription[] sources, String graph) {
            this.parent = parent;
            this.sources = sources;
            this.graph = graph;
        }

        public void run() {
            doIntegration(sources, graph);
            parent.taskEnded();
        }
    }

    public static void debug(String s) {
//        System.out.println("Debug IntegratorImpl: " + s);
    }
    
    /*
     * main method for test use only 
     */
    public static void main(String[] args) throws Exception {
        if (args.length < 2) {
            System.out.println("Give datasource id as first parameter & " +
                               "name of resourcegraph as second parameter.");
            return;
        }
        IntegratorImpl imp = new IntegratorImpl();
        imp.addGraph(args[1]);
        DataSourceDescription ds = new DataSourceDescription(
                args[0], "", "", 1, "", false);
        DataSourceDescription[] dss = new DataSourceDescription[1];
        dss[0] = ds;
        imp.integrate(dss, args[1]);
    }
}
