package maito.integration;

import maito.datacollecting.DataSourceDescription;
import maito.datacollecting.Record;
import maito.util.DbTools;

import java.sql.*;

/**
 * RecordReader.java
 *
 * @author Kalle Pyykkönen
 * @author OSaarekas
 * 
 * This software is released under the GNU GPL license
 */
public class RecordReader {
    
    private Connection conn;
    private Statement stmt;
    private ResultSet rs;
    private String currentSourceId;

    /**
     * Initializes Record reader.
     * 
     * @param con
     * Connection object to the database. The class will not work if
     * the object is invalid database connection.
     */
    public RecordReader(Connection con) {
        this.conn = con;
        this.stmt = null;
        this.rs = null;
        this.currentSourceId = null;
    }

    /**
     * Initialize this datasource to get records. If source parameter is changed
     * after last query executes a new query. 
     * Otherwise just resets the cursor to the first row.
     * 
     * @param source
     * DataSourceDescription object. 
     *
     */
    public void getRecords(DataSourceDescription source) {
        if (currentSourceId != null && source.getId().equals(currentSourceId)) {
            // set cursor to first row of a resultset
            try {
                if(rs != null) {
                    rs.first();
                }
            } catch (SQLException e) {
                e.printStackTrace();
            }
            return;
        }
        
        this.close();
        // new source, new query        
        currentSourceId = source.getId();

        try {
            stmt = conn.createStatement(ResultSet.CONCUR_READ_ONLY,
                    ResultSet.TYPE_SCROLL_INSENSITIVE);

            String query = "SELECT record, name, value, integratedTo " +
                    "FROM Statement, DataRecord WHERE " +
                    "Statement.record=DataRecord.id AND Statement.source=DataRecord.source " +
                    "AND DataRecord.source='" + DbTools.escSQL(source.getId()) + "'";
            rs = stmt.executeQuery(query);
            /*
             * Init cursor to ResultSet's first row,
             * or abandon if there are no rows
             */
            if (!rs.next()) {
                try {
                    rs.close();
                } catch (Exception e) {}
                rs = null;
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    /**
     * Gets the next record from the resultset. If ResultSet is at end, returns
     * null;
     * 
     * @return Record 
     * record object created from rs rows. If ResultSet is null or is at end,
     *         returns null.
     */
    public Record nextRecord() {
        if (rs == null) {
            return null;
        }

        try {
            if (rs.isAfterLast()) {
                return null;
            }

            Record rec = new Record(rs.getString("record"));
            //first set the special value from DataRecord 
            rec.setIntegratedTo(rs.getString("integratedTo"));
            //then the actual statements
            while (rs.getString("Record").equals(rec.getID())) {
                rec.setField(rs.getString("name"), rs.getString("value"));
                rs.next();
                if (rs.isAfterLast()) {
                    break;
                }
            }
            return rec;
        } catch (SQLException e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * Releases the objects for database handling
     */
    public void close() {
        if (rs != null) {
            try {
                rs.close();
            } catch (Exception e) {}
        }
        if (stmt != null) {
            try {
                stmt.close();
            } catch (Exception e) {}
        }
    }
}
