package maito.resource;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.sql.*;

import maito.util.DbTools;

/**
 * An object representing a complete resource in the resource
 * database. This object contains methods for loading & saving resources
 * to database & setting resource fields
 * 
 * @author Tuomas Tanner
 *
 */
public class Resource {
    
    public final static int IDLENGTH = 100;
    
    private String id;
    private String type;
    private HashMap data;
    private HashMap extraData;
    private HashMap records;
    private HashSet extIDs;
    private HashSet keyWords;
    
    private boolean loaded = false;
    private boolean overwrite = false;
    
    /**
     * Initialize this resource. After init the resource has only the 
     * id, type and created (current date) values.
     * 
     * The created field is overwritten if data is loaded to this resource.
     * 
     * @param id a unique id for this resource
     * @param type the type of resource this is. 
     * Allowed values: Actor, Document, Channel, Role
     */
    public Resource(String id, String type) {
        if (id == null) {
            id = "";
        }
        if (id.length() > IDLENGTH) {
            this.id = id.substring(0, IDLENGTH);
        }
        else {
            this.id = id;
        }
        this.type = type;
        this.data = new HashMap();
        this.data.put("created", DbTools.formatDate(new java.util.Date()));
        this.extraData = new HashMap();
        this.records = new HashMap();
        this.extIDs = new HashSet();
        this.keyWords = new HashSet();
    }
    
    /**
     * Set whether updates to this Record overwrite existing values or not.
     * If overwrite is true, existing values are overwritten. If false,
     * update will fail
     * @param which true of false
     */
    public void setOverwrite(boolean which) {
        this.overwrite = which;
    }
    
    /**
     *  Get current overwrite status
     * @return the status
     */
    public boolean getOverwrite() {
        return overwrite;
    }
    
    /**
     * Finds the unique id for this resource based on the current id.
     * If the current one is unique, all is well. If not we loop and append a counter
     * to the id (starts from 2).
     * 
     * @param con connection to database
     * @return the unique id that was found
     */
    public String findUniqueID(Connection con) {
        Statement stm = null;
        ResultSet rs = null;
        String tmpId = this.id;
        int appendix = 1;
        String query = "";
        try {
            stm = con.createStatement();
            query = "SELECT id FROM Resource WHERE id='" + DbTools.escSQL(tmpId) + "'";
            rs = stm.executeQuery(query);
            while (rs.next()) { //loop that adds 2,3,4 etc. to id until unique is found
                ++appendix;
                //check length (can't be longer than IDLENGTH)
                int overflow = (tmpId + appendix).length() - IDLENGTH;
                if (overflow > 0) {
                    tmpId = tmpId.substring(0, tmpId.length() - overflow);
                }
                query = "SELECT id FROM Resource WHERE id='" + DbTools.escSQL(tmpId + appendix) + "'";
                rs = stm.executeQuery(query);
            }
            if (appendix > 1) { //id has changed
                this.id = tmpId + appendix;
            }
            return this.id;
        } catch (SQLException e) {
            System.out.println("SQL ERROR (Resource.load): " + e.getMessage());
            System.out.println("Previous query: " + query);
            return null;
        } finally {
            if (rs != null)
                try {
                    rs.close();
                } catch (SQLException e) {}
            if (stm != null)
                try {
                    stm.close();
                } catch (SQLException e) {}
        }
    }
    
    public String getID() {
        return this.id;
    }
   
    /**
     * Sets the value for a specified key in this resource. The key must be specified
     * in the format "Tablename.Tablecolumn". The table name and column are case sensitive.
     * Both the table name and table column need to be correct - otherwise they are not set.
     * 
     * The Records field uses a special format: Key is "Records.source¤record" and 
     * value is "YYYY-MM-DD"
     * 
     * If overwrite is true the keys will be always set, if not they will only be set if
     * if they are missing. 
     * 
     * @param key Resource field to be saved
     * @param value the value for the field
     * @return
     */
    public boolean setField(String key, String value) {
        if (key == null) {
            return false;
        }
        int dot = key.indexOf(".");
        if (dot == -1) {
            return false;
        }
        String section = key.substring(0, dot);
        key = key.substring(dot + 1, key.length());
        
        if (section.equals("Resource")) {
            if (!key.equals("name") && !key.equals("canonicalName"))
                return false;
            if (overwrite || data.get(key) == null || data.get(key).equals("")) {
                data.put(key, value);
                return true;
            }
        }
        else if (section.equals("Role") || section.equals("Actor") ||
                 section.equals("Channel")) {
            if (!key.equals("type"))
                return false;
            if (overwrite || extraData.get(key) == null || extraData.get(key).equals("")) {
                extraData.put(key, value);
                return true;
            }
        }
        else if (section.equals("Document")) {
            if (!key.equals("referenceInfo") && !key.equals("published") 
                && !key.equals("language") && !key.equals("title"))
                return false;
            if (overwrite || extraData.get(key) == null || extraData.get(key).equals("")) {
                extraData.put(key, value);
                return true;
            }
        }
        else if (section.equals("KeyWords")) {
            if (!key.equals("keyword"))
                return false;
            keyWords.add(value); //always add - doesn't matter if same keyword overwritten
            return true;
        }
        else if (section.equals("ExtIDs")) {
            if (!key.equals("extID"))
                return false;
            extIDs.add(value); //always add - doesn't matter if same extID overwritten
            return true;
        }
        else if (section.equals("Records")) {
            if (overwrite || records.get(key) == null) {
                records.put(key, value);
                return true;
            }
        }
        return false;
    }
    
    /**
     * Loads the resource data from database into this Resource object.
     * 
     * @param con Connection to database 
     * @return true if resource was found and loaded successfully, false otherwise
     */
    public boolean load(Connection con) {
        Statement stm = null;
        ResultSet rs = null;
        String query = "";
        try {
            stm = con.createStatement();
            debug("loading resource:" + id);
            //load main values
            query = "SELECT * FROM Resource WHERE id=" + DbTools.safeStr(this.id);
            debug(query);
            rs = stm.executeQuery(query);
            if (!rs.next()) {
                return false;
            }
            ResultSetMetaData rsmd = rs.getMetaData();
            int colCount = rsmd.getColumnCount();
            for (int i = 1; i <= colCount; ++i) {
                String column = rsmd.getColumnName(i); 
                if (column.equals("id") || column.equals("type")) //skip special values
                    continue;
                if (rs.getString(i) != null) {
                    debug("put data: " + column + ", " + rs.getString(i));
                    data.put(column, rs.getString(i));
                }
            }
            //load other values
            query = "SELECT * FROM " + this.type + " WHERE id=" + DbTools.safeStr(this.id);
            debug(query);
            rs = stm.executeQuery(query);
            if (!rs.next()) {
                return false;
            }
            rsmd = rs.getMetaData();
            colCount = rsmd.getColumnCount();
            for (int i = 1; i <= colCount; ++i) {
                if (rs.getString(i) != null) {
                    debug("put extraData: " + rsmd.getColumnName(i) + ", " + rs.getString(i));
                    extraData.put(rsmd.getColumnName(i), rs.getString(i));
                }
            }
            //load Records
            query = "SELECT * FROM Records WHERE resource=" + DbTools.safeStr(this.id);
            debug(query);
            rs = stm.executeQuery(query);
            while (rs.next()) {
                debug("put records: " + rs.getString("source") + "," + rs.getString("record")
                        + "," + rs.getString("integrated"));
                records.put(rs.getString("source") + "¤" + rs.getString("record"), 
                        rs.getString("integrated"));
            }
            //load extIds
            query = "SELECT * FROM ExtIDs WHERE resource=" + DbTools.safeStr(this.id);
            debug(query);
            rs = stm.executeQuery(query);
            while (rs.next()) {
                debug("put extIDs:" + rs.getString("extID"));
                extIDs.add(rs.getString("extID"));
            }
            //load keywords
            query = "SELECT * FROM KeyWords WHERE resource=" + DbTools.safeStr(this.id);
            rs = stm.executeQuery(query);
            debug(query);
            while (rs.next()) {
                debug("put keyWords:" + rs.getString("keyword"));
                keyWords.add(rs.getString("keyword"));
            }
            this.loaded = true;
            
            return true;
        } catch (SQLException e) {
            System.out.println("SQL ERROR (Resource.load): " + e.getMessage());
            System.out.println("Previous query: " + query);
            return false;
        } finally {
            if (rs != null)
                try {
                    rs.close();
                } catch (SQLException e) {}
            if (stm != null)
                try {
                    stm.close();
                } catch (SQLException e) {}
        }
    }
    
        
    
    /**
     * Save this Resource object's state into the database
     * 
     * @param con
     * @return true if save was successfull, false otherwise
     */
    public boolean save(Connection con) {
        Statement stm = null;
        boolean autoCommit = true;
        String query = "";
        try {
            autoCommit = con.getAutoCommit();
            con.setAutoCommit(false);
            
            stm = con.createStatement();
            String prefix;
            String suffix;
            if (this.loaded) {
                prefix = "UPDATE ";
                suffix = " WHERE id=" + DbTools.safeStr(this.id);
            }
            else {
                prefix = "INSERT INTO ";
                suffix = "";
            }
            //save main resource
            Iterator iter = data.entrySet().iterator();
            String middle = "id=" + DbTools.safeStr(this.id) + 
                            ", type='" + this.type + "'";
            while (iter.hasNext()) {
                Map.Entry entry = (Map.Entry)iter.next();
                middle += "," + (String)entry.getKey() + 
                          "=" + DbTools.safeStr((String)entry.getValue());
            }
            
            query = prefix + "Resource SET " + middle + suffix;
            debug(query);
            stm.executeUpdate(query);
            
            //save additional resource info
            iter = extraData.entrySet().iterator();
            middle = "id=" + DbTools.safeStr(this.id);
            while (iter.hasNext()) {
                Map.Entry entry = (Map.Entry)iter.next();
                middle += "," + (String)entry.getKey() + 
                          "=" + DbTools.safeStr((String)entry.getValue());
            }
            query = prefix + this.type + " SET " + middle + suffix;
            debug(query);
            stm.executeUpdate(query);
            
            //(delete and) save the record infos
            if (this.loaded && records.size() > 0) {
                query = "DELETE FROM Records WHERE resource=" + 
                        DbTools.safeStr(this.id);
                debug(query);
                stm.executeUpdate(query);
            }
            iter = records.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry entry = (Map.Entry)iter.next();
                String[] keys = ((String)entry.getKey()).split("¤");
                query = "INSERT INTO Records SET resource=" + DbTools.safeStr(this.id) + 
                        ", source=" + DbTools.safeStr(keys[0]) + 
                        ", record=" + DbTools.safeStr(keys[1]) +
                        ", integrated=" + DbTools.safeStr((String)entry.getValue());
                debug(query);
                stm.executeUpdate(query);
            }
            
            //(delete and) save the KeyWords info
            if (this.loaded && keyWords.size() > 0) {
                query = "DELETE FROM KeyWords WHERE resource=" + 
                        DbTools.safeStr(this.id);
                debug(query);
                stm.executeUpdate(query);
            }
            iter = keyWords.iterator();
            while (iter.hasNext()) {
                String keyword = (String)iter.next();
                query = "INSERT INTO KeyWords SET resource=" + DbTools.safeStr(this.id) + 
                        ", keyword=" + DbTools.safeStr(keyword);
                debug(query);
                stm.executeUpdate(query);
            }
            
            //(delete and) save the extIDs info
            if (this.loaded && extIDs.size() > 0) {
                query = "DELETE FROM ExtIDs WHERE resource=" + 
                DbTools.safeStr(this.id);
                debug(query);
                stm.executeUpdate(query);
            }
            iter = extIDs.iterator();
            while (iter.hasNext()) {
                String extID = (String)iter.next();
                query = "INSERT INTO ExtIDs SET resource=" + DbTools.safeStr(this.id) + 
                        ", extID=" + DbTools.safeStr(extID);
                debug(query);
                stm.executeUpdate(query);
            }
            
            con.commit();
//            con.setAutoCommit(autoCommit);
            return true; //success
        } catch (SQLException e) {
            System.out.println("SQL ERROR-Resource.save(): " + e.getMessage());
            System.out.println("Previous query: " + query);
            return false;
        }
        finally {
            try {
                con.setAutoCommit(autoCommit);
            } catch(SQLException e) {}
            if (stm != null)
                try {
                    stm.close();
                } catch (SQLException e) {}
        }
    }
    
    public String getSingleField(String key) {
        if (key == null) {
            return null;
        }
        int dot = key.indexOf(".");
        if (dot == -1) {
            return null;
        }
        String section = key.substring(0, dot);
        key = key.substring(dot + 1, key.length());

        if (section.equals("Resource")) {
            return (String)data.get(key);
        }
        else if (section.equals("Role") || section.equals("Actor") ||
                 section.equals("Channel") || section.equals("Document")) {
            return (String)extraData.get(key);
        }
        else if (section.equals("Records")) {
            return (String)records.get(key);
        }
        return null;
    }

    public HashSet getKeywords() {
        return (HashSet)keyWords.clone();
    }

    public HashSet getExtIDs() {
        return (HashSet)extIDs.clone();
    }
    
    public String getType() {
        return this.type;
    }
    public StringBuffer toHTML() {
        
        StringBuffer html = new StringBuffer();
        
        html.append("<strong>Tunniste: </strong> ").append(this.id)
               .append(" <strong>Nimi: </strong>").append(this.getSingleField("Resource.name"))
               .append(" <strong>Kanonisoitu nimi: </strong>").append(this.getSingleField("Resource.canonicalName"));
        if (this.type.equals("Document")) {
            html.append(" <strong>Viittaustieto: </strong>").append(this.getSingleField("Document.referenceInfo"))
                        .append(" <strong>Julkaistu: </strong>").append(this.getSingleField("Document.published"))
                        .append(" <strong>Kieli: </strong>").append(this.getSingleField("Document.language"))
                        .append(" <strong>Otsikko: </strong>").append(this.getSingleField("Document.title"));
        }
        if (this.type.equals("Actor")) {
            html.append(" <strong>Tyyppi: </strong>").append(this.getSingleField("Actor.type"));
        }
        if (this.type.equals("Channel")) {
            html.append(" <strong>Tyyppi: </strong>").append(this.getSingleField("Channel.type"));
        }
        if (this.type.equals("Role")) {
            html.append(" <strong>Tyyppi: </strong>").append(this.getSingleField("Role.type"));
        }    
        return html;
    }
    
    
    private static void debug(String message) {
//        System.out.println("Debug Resource: " + message);
    }
}
