package maito.resource;

import java.sql.*;
import java.util.HashMap;

/**
 * This class acts as a buffered stream for reading Resource objects from the database.
 * @author Antti Laitinen
 *
 */
public class ResourceLoader {

    private Connection connection;

    /*The result of the latest query is stored here for buffering.
    We trust that the database driver handles that...*/
    private ResultSet resources;

    private HashMap extIDs;
    private HashMap Records;
    
    private Statement stmt;
    
    public ResourceLoader(Connection con) {

        if(con == null) {
            throw new IllegalArgumentException("ResourceLoader: no database connection provided");
        }

        this.connection = con;
        this.resources = null;
        this.stmt = null;
    }
    
    /**
     * Executes a query in the database. After this, the resources can be read using the
     * method nextResource().
     * @param sql
     * The sql to be executed. It is executed as it is, so any validation on the query
     * should be done outside this method.
     * @throws SQLException
     * Thrown when the executed query produces an error.
     */
    public void executeSQL(String sql) throws SQLException {
        this.stmt = this.connection.createStatement();
        this.resources = stmt.executeQuery(sql);
    }
    
    /**
     * Fetches the next resource produced by the latest query.
     * @return
     * A Resource object representing the resource in the database. null if the query hasn't
     * produced any resources or when all resources are read. 
     * @throws SQLException
     * Thrown if reading the database produces errors. 
     */
    public Resource nextResource() throws SQLException {
        
        if(this.resources == null) {
            return null;
        }
        
        Resource resource = null;
        
        if(this.resources.next()) {
            String resourceId = this.resources.getString("Resource.id");
            String resourceType = this.resources.getString("Resource.type");
            
            if(resourceId == null || resourceType == null) {
                return null;
            }
            
            resource = new Resource(resourceId, resourceType);
            
            if(!resource.load(this.connection)) {
                //TODO
            }
        }
        else {
            this.resources.close();
            this.resources = null;
            this.stmt.close();
        }
        
        return resource;
    }
}
