package maito.util;

import java.sql.*;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

/**
 * A class that contains methods for database related things.
 */
public class DbTools {

    private DbTools() {}
	
	public static Connection createDbConnection(Properties dbConfig) throws SQLException {
		String driverName = dbConfig.getProperty("jdbc_driver");
		String url = dbConfig.getProperty("jdbc_url");
		String user = dbConfig.getProperty("jdbc_user");
		String pass = dbConfig.getProperty("jdbc_pass");
        String conProps = dbConfig.getProperty("jdbc_conproperties");
        if (conProps == null) {
            conProps = "";
        }
        
		Connection con = null;
		
		if(driverName == null || url == null || user == null || pass == null) {
			throw new SQLException("connection parameters incomplete");
		}
		
		try {
			Class.forName(driverName).newInstance();
		}
		catch(Exception e) {
			throw new SQLException("Cannot load database driver: " + e.getMessage());
		}
		//sql exceptions are passed on
        url = url + "/" + conProps;
		con = DriverManager.getConnection(url, user, pass);
		return con;
	}
    
    public static Connection createDbConnection(Properties dbConfig, String dbName) throws SQLException {
        String driverName = dbConfig.getProperty("jdbc_driver");
        String url = dbConfig.getProperty("jdbc_url");
        String user = dbConfig.getProperty("jdbc_user");
        String pass = dbConfig.getProperty("jdbc_pass");
        String conProps = dbConfig.getProperty("jdbc_conproperties");
        if (conProps == null) {
            conProps = "";
        }
        
        Connection con = null;
        
        if(driverName == null || url == null || user == null || pass == null) {
            throw new SQLException("connection parameters incomplete");
        }
        if (dbName.length() > 0) {
            dbName = "/" + dbName;
        }
        
        try {
            Class.forName(driverName).newInstance();
        }
        catch(Exception e) {
            throw new SQLException("Cannot load database driver: " + e.getMessage());
        }
        //sql exceptions are passed on
        con = DriverManager.getConnection(url + dbName + conProps, user, pass);
        
        return con;
    }
    
    /**
     * Escapes some special SQL characters
     */
    public static String escSQL(String s) {
        if (s == null) {
            return null;
        }
        return s.replaceAll("'", "\\\\'");
    }
    /**
     * Escapes single quotes and returns the string quoted
     * with single quotes. null values create NULL -string
     * 
     */
    public static String safeStr(String s) {
        if (s == null) {
            return "NULL";
        }
        return "'" + s.replaceAll("'", "\\\\'") + "'";
    }
    
    /**
     * Formats a java.util.Date to the standard sql format YYYY-MM-DD
     * @param theDate the date to format
     * @return formatte date string
     */
    public static String formatDate(Date theDate) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        return df.format(theDate);
    }

    /**
     * Formats a string for SQL statement
     */
    public static String SQLstr(String s, int maxLen) {
        if (s == null) {
            s = "";
        }
        else if (!s.equals("")) {
            maxLen = Math.max(maxLen, 0);
            s = s.substring(0, Math.min(s.length(), maxLen));
        }
        
        return "'" + escSQL(s) + "'";
    }
}
