package maito.util;

import java.io.*;
import java.util.Vector;
import javax.xml.xpath.*;
import javax.xml.parsers.*;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.*;

/**
 * This class that holds a couple of useful methods for accessing xml data and writing an 
 * xml document to a file. This class is NOT an all purpose xml utility kit. The methods
 * that are included are designed only to be used by this program.
 * 
 * @author Antti Laitinen
 *
 */
public class XMLTools {
	
	private XMLTools() {}
	
	/**
	 * Returns the text content of an element in an XML document.
	 * @param builder
	 * Used in building a <code>org.w3c.dom.Document</code> object.
	 * @param xml
	 * Must be an InputStream which contains a valid xml document.
	 * @param xpathExpression
	 * An XPath expression which determines the target node in the document
	 * @return
	 * The content of the node as a string. <code>null</code> if the value or node is not available.
	 * @throws IOException
	 * @throws SAXException
	 * @throws XPathExpressionException
	 */
	public static String getNodeContent(DocumentBuilder builder, InputStream xml, String xpathExpression) throws IOException, SAXException, XPathExpressionException{
        
        String value = null;
        
        Document doc = builder.parse(xml);
        
        XPath xpath = XPathFactory.newInstance().newXPath();
        Node node = (Node) xpath.evaluate(xpathExpression, doc, XPathConstants.NODE);
        
        if(node != null) {
            NodeList children = node.getChildNodes();
            
            for ( int i = 0 ; i < children.getLength() ; i++ ) {
                if(children.item(i).getNodeType() == Node.TEXT_NODE) {
                    value = children.item(i).getNodeValue();
                }
                else if(children.item(i).getNodeType() == Node.CDATA_SECTION_NODE) {
                    value = children.item(i).getNodeValue();
                }
            }
        }
        
        return value;
    }
	
    /**
     * Returns the text content of one or more elements in an XML document.
     * If the given XPath expression matches more that one node then contents 
     * from all nodes are returned. 
     * @param builder
     * Used in building a <code>org.w3c.dom.Document</code> object.
     * @param xml
     * Must be an InputStream which contains a valid xml document.
     * @param xpathExpression
     * An XPath expression which determines the target node(s) in the document.
     * @return
     * The contents of all matching nodes as String[]. <code>null</code> if no match is found.
     * @throws IOException
     * @throws SAXException
     * @throws XPathExpressionException
     */
    public static String[] getAllContents(DocumentBuilder builder, InputStream xml, String xpathExpression) throws IOException, SAXException, XPathExpressionException{
        
        String[] contents = null;
        
        Vector v = new Vector();
        
        Document doc = builder.parse(xml);
        
        XPath xpath = XPathFactory.newInstance().newXPath();
        NodeList nodes = (NodeList) xpath.evaluate(xpathExpression, doc, XPathConstants.NODESET);
        
        for(int i = 0 ; i < nodes.getLength() ; i++) {
            
            Node node = nodes.item(i);
            
            if(node != null) {
                
                NodeList children = node.getChildNodes();
                
                for ( int j = 0 ; j < children.getLength() ; j++ ) {
                    if(children.item(j).getNodeType() == Node.TEXT_NODE) {
                        v.add(children.item(j).getNodeValue());
                    }
                    else if(children.item(j).getNodeType() == Node.CDATA_SECTION_NODE) {
                        v.add(children.item(j).getNodeValue());
                    }
                }
            }
        }
        
        contents = new String[v.size()];
        
        contents = (String[])v.toArray(contents);
        
        return contents;
    }
    
    /**
     * Writes a document to a file. NOTE: writing attributes is NOT supported! 
     * @param doc
     * @param filename
     * @return
     */
	public static File writeDocumentToFile(Document doc, String filename) {
		
		File file = null;
		
		org.w3c.dom.Element root = doc.getDocumentElement();
		String docString = nodeToString(root);
		
		try {
			file = new File(filename);
			if(!file.exists()) {
				file.createNewFile();
			}
			PrintWriter out = new PrintWriter(new FileOutputStream(file));
			out.print(docString);
			out.flush();
		}
		catch(Exception e) {
			e.printStackTrace();
			//TODO handle exception
		}
		
		return file;
	}
	
	
	private static String nodeToString(Node node) {
		NodeList children = node.getChildNodes();
		
		String nodeRepr = "";

		String startTag = "<" + node.getNodeName() + ">";
		String endTag = "</" + node.getNodeName() + ">";
		
		nodeRepr += startTag;
		for(int i = 0 ; i < children.getLength(); i++) {
			boolean childIsText = children.item(i).getNodeType() == Node.TEXT_NODE; 

			if(!childIsText) {
				nodeRepr += "\n" + nodeToString(children.item(i));
			}
			else {
				nodeRepr += children.item(i).getTextContent();
			}
				
		}
		nodeRepr += endTag + "\n";
				
		return nodeRepr;
	}
}
