package maito.datacollecting;

import junit.framework.TestCase;

import java.text.SimpleDateFormat;
import java.util.*;
import java.io.File;
import java.net.URL;
import java.sql.*;

import maito.util.DbTools;
import maito.util.Tools;

/**
 * This is a test driver for the class DataCollectorImpl.
 * 
 * NOTE: This test driver uses some test stubs that act as the OAI-PMH repository. 
 * The stubs are found at http://db.cs.helsinki.fi/~saarekas/stub/.
 * 
 * NOTE: There should be no existing data sources in the system.
 * 
 * @author Antti Laitinen
 *
 */
public class DataCollectorImplTest extends TestCase {

    private DataCollector collector;
    
    private Connection dbConnection;
    
    
    public DataCollectorImplTest() {
        this.collector = new DataCollectorImpl("build/data", "build/config");
    }
    
    
    public void setUp(){
        Properties dbSettings = Tools.loadProperties("build/config/dbconfig.properties");
        
        try {
            this.dbConnection = DbTools.createDbConnection(dbSettings,
                    dbSettings.getProperty("dbname_rawdata"));
            this.dbConnection.setAutoCommit(true);
        } catch (SQLException e) {
           e.printStackTrace();
        }
    }
    
    
    public void tearDown() {
        try {
            this.dbConnection.close();
        }
        catch(SQLException e) {
            e.printStackTrace();
        }
    }
    
    /**
     * Tests that when a source is added it can afterwards 
     * be seen using the method getSources() and it is written correctly in the database.
     *
     */
    public void testAddSource() {
        
        String url = "http://db.cs.helsinki.fi/~saarekas/stub/flow3.php";
        
        /*first, add the source*/
        try {
            this.collector.addSource("testsource", "oaipmh", 
                new URL(url),"DCXML");
        }
        catch(Exception e) {
            assertTrue(false);
        }
        
        /*test that the source is found using the getSources() method*/
        DataSourceDescription[] sources = this.collector.getSources();
        
        boolean sourceFound = false;
        
        for(int i = 0 ; i < sources.length ; i++) {
            if(sources[i].getId().equalsIgnoreCase(url)) {
                sourceFound = true;
            }
        }
        
        assertTrue(sourceFound);
        
        /*test that the source was written to the database*/
        
        try {
            Statement stmt = this.dbConnection.createStatement();
            ResultSet rs = stmt.executeQuery("select * from DataSource;");
            
            boolean idFound = false;
            
            while(rs.next()) {
            
                String id = rs.getString("id");
                
                if(id.equals(url)) {
                    idFound = true;
                    
                    String location = rs.getString("location");
                    assertTrue(location.equals(url));
                    String format = rs.getString("format");
                    assertTrue(format.equalsIgnoreCase("DCXML"));
                    assertTrue(rs.getString("updated") == null);
                    assertTrue(rs.getString("modified") == null);
                }
            }
            
            assertTrue(idFound);
        }
        catch(SQLException e) {
            e.printStackTrace();
            assertTrue(false);
        }
        
        this.collector.removeSources(this.collector.getSources(), true);
    }
    
    /**
     * Tests that all data sources are removed when using the removeSources() -method.
     *
     */
    public void testRemoveSources() {
        String url = "http://db.cs.helsinki.fi/~saarekas/stub/flow2.php";
        
        /*first, add a source*/
        try {
            this.collector.addSource("testsource", "oaipmh", 
                new URL(url),"DCXML");
        }
        catch(Exception e) {
            assertTrue(false);
        }
        
        this.collector.removeSources(this.collector.getSources(), true);
        
        /*check that the array returned by getSources() is empty*/
        assertTrue(this.collector.getSources().length == 0);
        
        /*check that the database is empty*/
        try {
            Statement stmt = this.dbConnection.createStatement();
            ResultSet rs = stmt.executeQuery("select * from DataSource;");
            assertTrue(!rs.next());
            rs = stmt.executeQuery("select * from DataRecord;");
            assertTrue(!rs.next());
            rs = stmt.executeQuery("select * from Statement;");
            assertTrue(!rs.next());
        }
        catch(Exception e) {
            assertTrue(false);
        }
    }
    
    
    /**
     * Tests that when a source is updated the data ends up in the database and 
     * the data source's field "updated" gets set in the database.
     *
     * This also tests at least someway that the workInProgress() method works because 
     * this method waits for the update to finish using that method.
     * 
     */
    public void testUpdate() {
        String url = "http://db.cs.helsinki.fi/~saarekas/stub/flow1.php";
        
        /*first, add a source*/
        try {
            this.collector.addSource("testsource", "oaipmh", 
                new URL(url),"DCXML");
        }
        catch(Exception e) {
            assertTrue(false);
        }
        
        /*update the source*/
        this.collector.updateSources(this.collector.getSources());
        
        while(this.collector.workInProgress()) {
          /*wait for the update to finish*/
            try {
                Thread.sleep(100);
            }
            catch(Exception e) {
                e.printStackTrace();
                assertTrue(false);
            }
        }
        
        try {
            Thread.sleep(1000);
        }
        catch(Exception e) {
            e.printStackTrace();
            assertTrue(false);
        }
        
        try {
            Statement stmt = this.dbConnection.createStatement();
            /*Make sure that there is data in the DataRecord table*/
            ResultSet rs = stmt.executeQuery("select * from DataRecord;");
            assertTrue(rs.next());
            /*Make sure that there is data in the Statement table*/
            rs = stmt.executeQuery("select * from Statement;");
            assertTrue(rs.next());
            
            /*make sure that the updated field is set to the current date
             * in the format yyyy-MM-dd*/
            
            String now = new SimpleDateFormat("yyyy-MM-dd").format(new java.util.Date());
            
            rs = stmt.executeQuery("select updated from DataSource where location='" + url + "';");
            rs.next();
            assertTrue(rs.getString("updated") != null);
            assertTrue(rs.getString("updated").equals(now));
            rs.close();
            
        }
        catch(Exception e) {
            e.printStackTrace();
            assertTrue(false);
        }
        
        System.out.println("removing sources");
        
        this.collector.removeSources(this.collector.getSources(), true);
        
        assertTrue(this.collector.getSources().length == 0);
        
    }
}
