/*
* ListRecordsTest.java
* 
* v0.1
*
* 10.11.2005
* 
* This software is released under the GNU GPL license
*/
package maito.datacollecting.oaipmh;

import junit.framework.TestCase;

import java.net.*;
import java.io.*;
import java.util.regex.*;

/**
 * The main target of this test driver is the class maito.datacollecting.oaipmh.ListRecords. Also
 * maito.datacollecting.oaipmh.ListRecordsContentHandler can be seen as a target for this test.
 * This test covers 100% of the code in that class.
 *   
 * NOTE: This test driver uses some test stubs that act as the OAI-PMH repository. 
 * The stubs are found at http://db.cs.helsinki.fi/~saarekas/stub/.
 * 
 * @author Antti Laitinen
 *
 */
public class ListRecordsTest extends TestCase {

	private ListRecords listRecords;

	private ServerSocket serverSocket;
	
	/*edit this field so that it contains a port which can be listened to on your system*/
	private final int SERVER_PORT = 11111;
	
	private ServerThread server;
	
	
	public ListRecordsTest(String name) {
		super(name);
		
	}
	
	private void setUpServer() {
		try {
			this.serverSocket = new ServerSocket(SERVER_PORT);
		}
		catch(IOException e) {
			e.printStackTrace();
		}
		
		//start a new thread that listens to incoming calls 
		this.server = new ServerThread(this.serverSocket);
		this.server.start();
		
	}
	

	/**
	 * Tests that ListRecords behaves correctly when the parameters passed to it's constructor are incorrect.
	 * ListRecords should only throw OAIPMHException.
	 *
	 */
	public void testBadConstructorParameters() {
		
		//all params are null
		try {
			 listRecords = new ListRecords(null, null, null);
			 assertTrue(false);
		}
		catch(Exception e) {
			assertTrue(e instanceof OAIPMHException);
		}
		
		//metadataPrefix is null
		try {
			 listRecords = new ListRecords("http://repository.org/oai", null, null);
			 assertTrue(false);
		}
		catch(Exception e) {
			assertTrue(e instanceof OAIPMHException);
		}
		
		//baseURL is null
		try {
			 listRecords = new ListRecords(null, "oai_dc", null);
			 assertTrue(false);
		}
		catch(Exception e) {
			assertTrue(e instanceof OAIPMHException);
		}
		
		//baseURL is malformed
		try {
			 listRecords = new ListRecords("foobar", "oai_dc", null);
			 assertTrue(false);
		}
		catch(Exception e) {
			assertTrue(e instanceof OAIPMHException);
		}
	}
	
	
	/**
	 * Tests that the dateFrom parameter is passed in the request unchanged.
	 *
	 */
	public void testDateFrom() {
		
		String dateFrom = "2005-10-09";
		
		this.setUpServer();
		
		try {
			 listRecords = new ListRecords("http://localhost:" + SERVER_PORT, "oai_dc", dateFrom);
		}
		catch(Exception e) {
		}
		
		String request = this.server.getRequest();
		
		Pattern dateFromPattern = Pattern.compile("from=" + dateFrom);
		Matcher matcher = dateFromPattern.matcher(request);
		
		assertTrue(matcher.find());
	
	}
	
	/**
	 * Tests reading a huge part at the time from the response.
	 *
	 */
	public void testHugePartSize() {
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow1.php", "oai_dc", null);
		}
		catch(Exception e) {
			e.printStackTrace();
			assertTrue(false);
		}
		try {
			 String part = listRecords.nextPart(100000);
			 while(part != null) {
				 part = listRecords.nextPart(100000);
			 }
		}
		catch(Exception e) {
			assertTrue(false);
		}

	}
	
	/**
	 * Tests reading the smallest possible part at the time from the response.
	 *
	 */
	
	public void testSmallPartSize() {
	
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow1.php", "oai_dc", null);
		}
		catch(Exception e) {
			assertTrue(false);
		}
		try {
			 String part = listRecords.nextPart(1);
			 
			 while(part != null) {
				 part = listRecords.nextPart(1);
			 }
		}
		catch(Exception e) {
			assertTrue(false);
		}
	}
	
	/**
	 * Tests that ListRecords behaves correctly when asked for a negative amount of characters.
	 *
	 */
	public void testAskNegativeAmount() {
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow1.php", "oai_dc", null);
		}
		catch(Exception e) {
			assertTrue(false);
		}
		try {
			 assertTrue(listRecords.nextPart(-1) == null);
		}
		catch(Exception e) {
			assertTrue(false);
		}
	}
    
    /**
     * Tests that ListRecords behaves correctly when asked for zero characters.
     *
     */
    public void testAskZeroAmount() {
        try {
             listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow1.php", "oai_dc", null);
        }
        catch(Exception e) {
            assertTrue(false);
        }
        try {
             assertTrue(listRecords.nextPart(0) == null);
        }
        catch(Exception e) {
            assertTrue(false);
        }
    }
	
	/**
	 * Tests that ListRecords behaves correctly when the OAIPMH response contains an error.
	 * The test stub must return an OAI-PMH error at some point.
	 *
	 */
	public void testOAIPMHError() {
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow2.php", "oai_dc", null);
		}
		catch(Exception e) {
			assertTrue(false);
		}
		try {
			 String part = listRecords.nextPart(10);
			 
			 while(part != null) {
				 part = listRecords.nextPart(1);
			 }
             
			 //we get here if no exception was found even when an OAI-PMH error was received
			 assertTrue(false);
		}
		catch(Exception e) {
			assertTrue(e instanceof OAIPMHException);
		}
	}
	
	/**
	 * Tests that ListRecords behaves correctly when the response is an invalid xml.
	 */
	public void testInvalidXML() {
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/invalidxml.php", "oai_dc", null);
		}
		catch(Exception e) {
			assertTrue(e instanceof OAIPMHException);
		}
	}

	/**
	 * Tests that the received response contains only record tags from the OAI-PMH response.
	 *
	 */
	
	public void testRecordTagsOnly() {
		
		StringBuffer response = new StringBuffer();
		
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow1.php", "oai_dc", null);
			 
			 String part = listRecords.nextPart(100);
			 while(part != null) {
				 response.append(part);
				 part = listRecords.nextPart(100);
			 }
		}
		catch(Exception e) {
			assertTrue(false);
		}
				
		String startTag = "<record>";
		String endTag = "</record>";
		
		int startIndex = response.indexOf(startTag); 
		int endIndex = response.indexOf(endTag);
		
		while(startIndex >= 0 && endIndex >= 0) {
			response = response.delete(startIndex, endIndex + endTag.length());
			startIndex = response.indexOf(startTag);
			endIndex = response.indexOf(endTag);
		}
		
		assertTrue(response.length() == 0);
	}
	
	/**
	 * Tests flow control. The stub that is used should result in 3 record elements if the flow control works in ListRecords.
	 *
	 */
	public void testFlowControl() {
		int recordAmount = 0;
		
		StringBuffer response = new StringBuffer();
		
		try {
			 listRecords = new ListRecords("http://db.cs.helsinki.fi/~saarekas/stub/flow1.php", "oai_dc", null);
			 
			 String part = listRecords.nextPart(100);
			 while(part != null) {
				 response.append(part);
				 part = listRecords.nextPart(100);
			 }
		}
		catch(Exception e) {
			assertTrue(false);
		}
		
		String startTag = "<record>";
		String endTag = "</record>";
		
		int startIndex = response.indexOf(startTag); 
		int endIndex = response.indexOf(endTag);
		
		while(startIndex >= 0 && endIndex >= 0) {
			recordAmount++;
			response = response.delete(startIndex, endIndex + endTag.length());
			startIndex = response.indexOf(startTag);
			endIndex = response.indexOf(endTag);
		}
		
		assertTrue(recordAmount == 3);
	}
	
    /**
     * Tests that ListRecords behaves correctly when one of the server's responses
     * is an unparseable xml. The xml should be omitted and harvesting should continue
     * if a resumptionToken is still found from the data.  
     *
     * The test stub that is used here should return 3 xml documents. 
     * The second should be invalid and the harvesting should result in 2 records.
     */
    public void testErrorHandling() {
       
        final String TEST_STUB = "http://db.cs.helsinki.fi/~saarekas/stub/flow4.php";
        
        int recordAmount = 0;
        
        StringBuffer response = new StringBuffer();
        
        try {
             listRecords = new ListRecords(TEST_STUB, "oai_dc", null);
             
             String part = listRecords.nextPart(100);
             while(part != null) {
                 response.append(part);
                 part = listRecords.nextPart(100);
             }
        }
        catch(Exception e) {
            assertTrue(false);
        }
        
        String startTag = "<record>";
        String endTag = "</record>";
        
        int startIndex = response.indexOf(startTag); 
        int endIndex = response.indexOf(endTag);
        
        while(startIndex >= 0 && endIndex >= 0) {
            recordAmount++;
            response = response.delete(startIndex, endIndex + endTag.length());
            startIndex = response.indexOf(startTag);
            endIndex = response.indexOf(endTag);
        }
        
        assertTrue(recordAmount == 2);
        
    }
    
	
	/**
	 * A class for accepting connections to a server socket and reading the request.
	 * @author Antti Laitinen
	 *
	 */
	class ServerThread extends Thread {
		
		private ServerSocket socket;
		
		private StringBuffer request;
		
		
		public ServerThread(ServerSocket socket) {
			this.socket = socket;
			this.request = null;
		}
		
		public void run() {
			try {
				
				Socket sock = this.socket.accept();
				
				
				
				/*reads only the request and after that closes the connection*/
								
				BufferedReader reader = new BufferedReader(new InputStreamReader(sock.getInputStream()));
				
				this.request = new StringBuffer();
				
				String line = reader.readLine();
				
				while(line != null) {
					this.request.append(line);
					line = reader.readLine();
					if(line.equals("")) {
						break;
					}
				}
				
				sock.close();
				
				this.socket.close();
				
			}
			catch(IOException e) {
				e.printStackTrace();
				System.exit(1);
			}
		}
		
		public String getRequest() {
			return this.request.toString();
		}
	}
}
