package maito.datacollecting.oaipmh;

import junit.framework.TestCase;

import maito.datacollecting.RecordParser;
import maito.datacollecting.*;
import maito.util.Tools;

import java.util.Properties;
import java.util.regex.*;

/**
 * 
 * This class implements the RecordParser interface so that OAIPMHDataSource can pass it's
 * data here.
 * 
 * NOTE: This test driver uses some test stubs that act as the OAI-PMH repository. 
 * The stubs are found at http://db.cs.helsinki.fi/~saarekas/stub/.
 * 
 * @author Antti Laitinen
 *
 */
public class OAIPMHDataSourceTest extends TestCase implements RecordParser{

    private OAIPMHDataSource source;
    
    private StringBuffer data;
    
    
    public OAIPMHDataSourceTest(String name) {
        super(name);
        this.data = new StringBuffer();
    }

    
    /**
     * Tests that the data is passed to the RecordParser (this object).
     * The RecordParser should receive a number of record sections defined
     * by REQUIRED_RECORD_AMOUNT.
     *
     */
    public void testDataFlow() {
        
        final int REQUIRED_RECORD_AMOUNT = 3;
        final String TEST_STUB = "http://db.cs.helsinki.fi/~saarekas/stub/flow1.php";
        
        Properties params = new Properties();
        
        params.put(Tools.DATASOURCE_PARAM_LOCATION, TEST_STUB);
        
        try {
            this.source = new OAIPMHDataSource(params, this);
            this.source.update();
        }
        catch(DataSourceException e) {
            e.printStackTrace();
            assertTrue(false);
        }
        
        Pattern pattern = Pattern.compile("<record>");
        Matcher matcher = pattern.matcher(this.data);
        
        int recordsFound = 0;
        
        while(matcher.find()) {
            recordsFound++;
        }
        
        assertTrue(recordsFound == REQUIRED_RECORD_AMOUNT);
    }

    /**
     * Tests that OAIPMHDataSource behaves correctly when the parameters are invalid.
     * It should only throw DataSourceException.
     */
    public void testInvalidParameters() {
        
        final String TEST_STUB = "http://db.cs.helsinki.fi/~saarekas/stub/flow1.php";
        
        /*leave parameters empty*/
        Properties params = new Properties();
        
        try {
            this.source = new OAIPMHDataSource(params, this);
            assertTrue(false);
        }
        catch(Exception e) {
            assertTrue(e instanceof DataSourceException);
        }
        
        /*params is null*/
        try {
            this.source = new OAIPMHDataSource(null, this);
            assertTrue(false);
        }
        catch(Exception e) {
            assertTrue(e instanceof DataSourceException);
        }
        
        /*RecordParser is null*/
        params = new Properties();
        params.put(Tools.DATASOURCE_PARAM_LOCATION, TEST_STUB);
        
        try {
            this.source = new OAIPMHDataSource(params, null);
            assertTrue(false);
        }
        catch(Exception e) {
            assertTrue(e instanceof DataSourceException);
        }
        
        /*all parameters are null*/
        try {
            this.source = new OAIPMHDataSource(null, null);
            assertTrue(false);
        }
        catch(Exception e) {
            assertTrue(e instanceof DataSourceException);
        }
    }
    
    /**
     * Tests that OAIPMHDataSource behaves correctly when the repository sends
     * invalid data.
     *
     */
    public void testInvalidRepository() {
        
        final String TEST_STUB = "http://db.cs.helsinki.fi/~saarekas/stub/invalidxml.php";
        
        /*leave parameters empty*/
        Properties params = new Properties();
        params.put(Tools.DATASOURCE_PARAM_LOCATION, TEST_STUB);        
        
        try {
            this.source = new OAIPMHDataSource(params, this);
            this.source.update();
            assertTrue(false);
        }
        catch(Exception e) {
            assertTrue(e instanceof DataSourceException);
        }
    }
    
    /**
     * Tests that OAIPMHDataSource behaves correctly when an error occurs somewhere in
     * the middle of the data stream.
     *
     */
    public void testErrorMidStream() {
        
        final String TEST_STUB = "http://db.cs.helsinki.fi/~saarekas/stub/flow2.php";
        
        /*leave parameters empty*/
        Properties params = new Properties();
        params.put(Tools.DATASOURCE_PARAM_LOCATION, TEST_STUB);        
        
        try {
            this.source = new OAIPMHDataSource(params, this);
            this.source.update();
            assertTrue(false);
        }
        catch(Exception e) {
            assertTrue(e instanceof DataSourceException);
        }
    }
    
    
    public void putData(String data) {
            this.data.append(data);
    }

}
