/**
* QuickformatTransformerTest
* 
*
* 28.11.2005
* 
* This software is released under the GNU GPL license
* 
* @version 3.0
* @author Reima Halmetoja
*/
package maito.datacollecting.quickformat;

import java.util.Vector;

import maito.datacollecting.Record;
import junit.framework.TestCase;

public class QuickformatTransformerTest extends TestCase {

		
    /*
     * Test method for 'maito.datacollecting.quickformat.QuickformatTransformer.transform(Record)'
     * 
     * Tests a Record with only document-type quickformat.
     */
    public void testDocumentTransform() {

        QuickformatTransformer testTfm = new QuickformatTransformer();
                
        // Create a test record...
        Record testRec = new Record("testi");
        testRec.setField("documentqf:identifier", "testi1¤testi2");
        testRec.setField("documentqf:creator", "Shapiro, B.¤Kazarian, M. E.");
        testRec.setField("documentqf:title", "Giambelli-type formula for subbundles of the tangent bundle");
        testRec.setField("documentqf:publisher", "ACM¤Association for Computing Machinery");
        testRec.setField("documentqf:publication", "ACM Transactions on Database Systems");
        testRec.setField("documentqf:publishedYear","2004-08-13");
        testRec.setField("documentqf:referenceIdentifier", "PRKL666");
        testRec.setField("documentqf:isReferencedBy", "HejsanSverige1");
        testRec.setField("documentqf:references", "Joku dokkari");
        testRec.setField("documentqf:keywords","test¤testcase¤testiTapausNumero1234");
   
        // Then transform it...
        Record resultRec = new Record("foo");
        try {
            resultRec = testTfm.transform(testRec);
        }
        catch (Exception e) {
        	assertTrue(false);
        }
        
        Vector temp;
        assertTrue(resultRec.getID().equals("testi"));
        
        temp = resultRec.getField("SisäinenTunniste");
        assertTrue(temp.size() == 2); // two values separated by ¤
        assertTrue(temp.elementAt(0).equals("testi1"));
        assertTrue(temp.elementAt(1).equals("testi2"));
        
        temp = resultRec.getField("HenkilöTekijä");
        assertTrue(temp.size() == 2);
        assertTrue(temp.elementAt(0).equals("Shapiro, B."));
        assertTrue(temp.elementAt(1).equals("Kazarian, M. E."));
        
        temp = resultRec.getField("Nimeke");
        assertTrue(temp.size() == 1);
        assertTrue(temp.firstElement().equals("Giambelli-type formula for subbundles of the tangent bundle"));

        temp = resultRec.getField("JokuJulkaisija");
        assertTrue(temp.size() == 2); // two values separated by '¤'.
        assertTrue(temp.elementAt(0).equals("ACM"));
        assertTrue(temp.elementAt(1).equals("Association for Computing Machinery"));
        
        temp = resultRec.getField("JulkaisuNimi");
        assertTrue(temp.size() == 1);
        assertTrue(temp.firstElement().equals("ACM Transactions on Database Systems"));
                
        temp = resultRec.getField("Julkaisupäivämäärä");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("2004-08-13"));

        temp = resultRec.getField("Viittaustunniste");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("PRKL666"));
        
        temp = resultRec.getField("Viitattu");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("HejsanSverige1"));
  
        temp = resultRec.getField("Viittaava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Joku dokkari"));
        
        temp = resultRec.getField("Hakusanat");
        assertTrue(temp.size() == 3);
        assertTrue(temp.elementAt(0).equals("TEST"));
        assertTrue(temp.elementAt(1).equals("TESTCASE"));
        assertTrue(temp.elementAt(2).equals("TESTITAPAUSNUMERO1234"));
    }
    
    /*
     * Test method for 'maito.datacollecting.quickformat.QuickformatTransformer.transform(Record)'
     * 
     * Tests a Record name-type quickformat. The fields are not in a configuration that would
     * occur in real use.
     */
    public void testNameTransform() {

        QuickformatTransformer testTfm = new QuickformatTransformer();
        
        Record testRec = new Record("testi");
        testRec.setField("nameqf:actorUnknown", "Some Actor");
        testRec.setField("nameqf:person", "Yeltsin, B.");
        testRec.setField("nameqf:organization", "McGraw-Hill");
        testRec.setField("nameqf:roleUnknown", "Random House, Inc.¤Random House");
        testRec.setField("nameqf:publisher", "Macmillan¤Macmillan Publishers, Ltd.¤Macmillan Technical Publishing");

        testRec.setField("nameqf:channelUnknown","tuntematon kanava");
        testRec.setField("nameqf:magazine", "IEICE - Transactions on Fundamentals of Electronics, Communications and Computer Sciences|Algorithms,Computational Theory,joku kolmas vielä");
        testRec.setField("nameqf:conference", "ACM-SE¤ACM Southeast Regional Conference: ACM Press");
        testRec.setField("nameqf:publicationSet", "Joku julkaisusarja: tähänkin kaksoispiste");
        testRec.setField("nameqf:reportSet","Joku raporttisarja");
        testRec.setField("nameqf:database","datatukikohta2005");
        
        // Then transform it...
        Record resultRec = new Record("foo");
        try {
            resultRec = testTfm.transform(testRec);
        }
        catch (Exception e) {
        	assertTrue(false);
        }
        
        Vector temp;
        
        //Some elements in the Vector with the same field name
        
        // parts before "¤".
        temp = resultRec.getField("TuntematonToimija");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Some Actor"));

        temp = resultRec.getField("HenkilöToimija");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Yeltsin, B."));
       
        temp = resultRec.getField("OrganisaatioToimija");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("McGraw-Hill"));
        
        temp = resultRec.getField("JokuRooli");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Random House, Inc."));
        
        temp = resultRec.getField("JokuJulkaisija");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Macmillan"));
        
        //Additional entries separated by "¤".
        //Note that the Record will not store the entries in the order they are entered in.
        temp = resultRec.getField("SisäinenTunniste");
        assertTrue(temp.size() == 4);
        assertTrue(temp.elementAt(2).equals("RANDOM HOUSE"));
        assertTrue(temp.elementAt(0).equals("MACMILLAN PUBLISHERS, LTD."));
        assertTrue(temp.elementAt(1).equals("MACMILLAN TECHNICAL PUBLISHING"));
        assertTrue(temp.elementAt(3).equals("ACM SOUTHEAST REGIONAL CONFERENCE"));
                     
        temp = resultRec.getField("JokuKanava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("tuntematon kanava"));
        
        temp = resultRec.getField("LehtiKanava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("IEICE - Transactions on Fundamentals of Electronics, Communications and Computer Sciences"));
        
        temp = resultRec.getField("KonferenssiKanava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("ACM-SE"));
        
        temp = resultRec.getField("JulkaisusarjaKanava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Joku julkaisusarja"));
        
        temp = resultRec.getField("RaporttisarjaKanava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("Joku raporttisarja"));
        
        temp = resultRec.getField("TietokantaKanava");
        assertTrue(temp.size() == 1);
        assertTrue(temp.elementAt(0).equals("datatukikohta2005"));

        
        //Entries separated by colon (": ")
        temp = resultRec.getField("OrganisaatioHallinnoija");
        assertTrue(temp.size() == 2);
        assertTrue(temp.elementAt(1).equals("ACM Press"));          
        assertTrue(temp.elementAt(0).equals("tähänkin kaksoispiste"));  
        
        //Entries separated by pipe ("|") and commas (",").
        temp = resultRec.getField("Aihe");
        assertTrue(temp.size() == 3);
        assertTrue(temp.elementAt(0).equals("ALGORITHMS"));
        assertTrue(temp.elementAt(1).equals("COMPUTATIONAL THEORY"));
        assertTrue(temp.elementAt(2).equals("JOKU KOLMAS VIEL"));
    }
    
    /**
	 * Finally a null record should cause an exception.
	 *
	 */

    public void testNullTransform() {	
        QuickformatTransformer testTfm = new QuickformatTransformer();

        Record nullRec = null;
        try {
            testTfm.transform(nullRec);
            assertTrue(false); //shouldn't get here
        }
        catch (Exception e) {
            assertTrue(true);
        }
    }

}
