/*
 * IntegratorHelperTest.java
 *
 * This software is released under the GNU GPL license
 *
 */

package maito.integration;

import java.io.File;
import java.sql.*;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;

import maito.util.Tools;
import maito.util.DbTools;
import maito.resource.Resource;
import maito.datacollecting.Record;

import junit.framework.TestCase;

public class IntegratorHelperTest extends TestCase {
    private Connection conn = null;
    private Statement stmt = null;
    private IntegratorHelper ih = null;

	public IntegratorHelperTest(String name) throws Exception {
		super(name);
		Class.forName("com.mysql.jdbc.Driver").newInstance();

        // load document fieldmap file
        Properties docFieldmap = Tools.loadProperties("config" + File.separator + "fieldmap_document.properties");
        if (docFieldmap == null) {
            System.out.println("ERROR: Could not load fieldmap_document.properties");
        }
        // load actor & channel fieldmap file
        Properties actorFieldmap = Tools.loadProperties("config" + File.separator + "fieldmap_actorchannel.properties");
        if (actorFieldmap == null) {
            System.out.println("ERROR: Could not load fieldmap_document.properties");
        }
        // load quicknameformat fieldmap file
        Properties quickNameFieldmap = Tools.loadProperties("config" + File.separator + "fieldmap_qnformat.properties");
        if (quickNameFieldmap == null) {
            System.out.println("ERROR: Could not load fieldmap_qnformat.properties");
        }
        
        this.ih = new IntegratorHelper(docFieldmap, actorFieldmap, quickNameFieldmap);
	}
	
	public void setUp() throws Exception {
		conn = DriverManager.getConnection(
	        "jdbc:mysql://localhost/maitotest?user=maito&password=maito");
		assertTrue(conn != null);
		stmt = conn.createStatement();
    }
    
    public void tearDown() throws Exception {
        stmt = null;
        conn.close();
    }

    private void resetDB() throws Exception {        
        stmt.execute("TRUNCATE TABLE resourcerelation;");
        stmt.execute("TRUNCATE TABLE actor;");
        stmt.execute("TRUNCATE TABLE channel;");
        stmt.execute("TRUNCATE TABLE document;");
        stmt.execute("TRUNCATE TABLE extids;");
        stmt.execute("TRUNCATE TABLE keywords;");
        stmt.execute("TRUNCATE TABLE records;");
        stmt.execute("DELETE FROM role WHERE NOT id='NONE' AND NOT id='BIBREF';");
        stmt.execute("DELETE FROM resource WHERE NOT id='NONE' AND NOT id='BIBREF';");
    }

    private int numRelations(String s, String o, String r) throws SQLException {
        ResultSet rs = null;
        int ret = 0;
        if (r.equals("NONE") || r.equals("BIBREF")) {
            rs = stmt.executeQuery("SELECT COUNT(role) FROM ResourceRelation WHERE role='" +
                                   r+"' AND subject='"+s+"' AND object='"+o+"'");
            if (rs.next()) {
                ret = rs.getInt(1);
            }
            rs.close();
            stmt.execute("DELETE FROM ResourceRelation WHERE role='"+r+
                         "' AND subject='"+s+"' AND object='"+o+"'");
        }
        else {
            rs = stmt.executeQuery("SELECT COUNT(id) FROM ResourceRelation, Role WHERE " +
                                   "ResourceRelation.role=Role.id AND Role.type='" +
                                   r+"' AND subject='"+s+"' AND object='"+o+"'");
            if (rs.next()) {
                ret = rs.getInt(1);
            }
            rs.close();
            stmt.execute("DELETE FROM ResourceRelation WHERE subject='"+s+
                "' AND object='"+o+"' AND role=ANY(SELECT id FROM Role WHERE type='"+r+"')");
        }
        return ret;
    }
    
    private int numRoles(String t) throws SQLException {
        ResultSet rs = null;
        int ret = 0;
        rs = stmt.executeQuery("SELECT COUNT(id) FROM Role WHERE type='"+t+"'");
        if (rs.next()) {
            ret = rs.getInt(1);
        }
        rs.close();
        stmt.execute("DELETE FROM Role WHERE type='"+t+"'");
        return ret;
    }
    
    private int numActors(String n) throws SQLException {
        ResultSet rs = null;
        int ret = 0;
        rs = stmt.executeQuery("SELECT COUNT(Actor.id) FROM Resource, Actor WHERE " +
                               "Resource.id=Actor.id AND Resource.name='"+n+"'");
        if (rs.next()) {
            ret = rs.getInt(1);
        }
        rs.close();
        stmt.execute("DELETE FROM Resource WHERE Resource.name='"+n+
                     "' AND id=ANY(SELECT id FROM Actor)");
        return ret;
    }
    
    private int numChannels(String n) throws SQLException {
        ResultSet rs = null;
        int ret = 0;
        rs = stmt.executeQuery("SELECT COUNT(Channel.id) FROM Resource, Channel WHERE " +
                               "Resource.id=Channel.id AND Resource.name='"+n+"'");
        if (rs.next()) {
            ret = rs.getInt(1);
        }
        rs.close();
        stmt.execute("DELETE FROM Resource WHERE Resource.name='"+n+
                     "' AND id=ANY(SELECT id FROM Channel)");
        return ret;
    }
    
    private boolean tableIsEmpty(String t) throws SQLException {
        return !(stmt.executeQuery("SELECT * FROM " + t).next());
    }
    
    private boolean roleTableIsEmpty() throws SQLException {
        return !(stmt.executeQuery("SELECT * FROM Role WHERE NOT id='NONE' AND NOT id='BIBREF'").next());
    }
    
    
    /*
     * Actual test methods begin
     */
    
    public void testGetFirstVal() throws Exception {
        assertTrue(IntegratorHelper.getFirstVal(null) == null);
        Vector v = new Vector();
        assertTrue(IntegratorHelper.getFirstVal(v) == null);
        v.add("Class 52");
        v.add("1012");
        assertTrue(IntegratorHelper.getFirstVal(v).equals("Class 52"));
    }

    public void testCreateDocID() throws Exception {
        Record r = null;
        String s = null;
        
        r = new Record("id");
        s = IntegratorHelper.createDocID(r);
        assertTrue(s.length() == 6);
        for (int i = 0; i < 6; ++i) {
            assertTrue((s.charAt(i) - 'a') >= 0 && (s.charAt(i) - 'a') <= 25);
        }
        r.setField("HenkilöTekijä", "ab");
        s = IntegratorHelper.createDocID(r);
        assertTrue(s.length() == 6);
        for (int i = 0; i < 6; ++i) {
            assertTrue((s.charAt(i) - 'a') >= 0 && (s.charAt(i) - 'a') <= 25);
        }
        r.deleteField("HenkilöTekijä");
        
        r.setField("JokuTekijä", "abc");
        assertTrue(IntegratorHelper.createDocID(r).equals("abcXXxxxx"));
        r.setField("OrganisaatioTekijä", "bcd");
        assertTrue(IntegratorHelper.createDocID(r).equals("bcdXXxxxx"));
        r.setField("HenkilöTekijä", "cde");
        assertTrue(IntegratorHelper.createDocID(r).equals("cdeXXxxxx"));
        
        r.setField("Julkaisupäivämäärä", "199");
        assertTrue(IntegratorHelper.createDocID(r).equals("cdeXXxxxx"));
        r.deleteField("Julkaisupäivämäärä");
        r.setField("Julkaisupäivämäärä", "1995");
        assertTrue(IntegratorHelper.createDocID(r).equals("cde95xxxx"));
        
        r.setField("Nimeke", "Moi");
        assertTrue(IntegratorHelper.createDocID(r).equals("cde95xxxx"));
        r.deleteField("Nimeke");
        r.setField("Nimeke", "Moi Hei");
        assertTrue(IntegratorHelper.createDocID(r).equals("cde95xxxx"));
        r.deleteField("Nimeke");
        r.setField("Nimeke", "Moi Heippa");
        assertTrue(IntegratorHelper.createDocID(r).equals("cde95Heippa"));
        r.deleteField("Nimeke");
        r.setField("Nimeke", "Moi!");
        assertTrue(IntegratorHelper.createDocID(r).equals("cde95Moi!"));
    }
    
    public void testGetResource() throws Exception {
        resetDB();
        Resource r = null;
        
        r = ih.getResource(conn, "Actor", "Organisaatio", "Name1", "NAME1");
        r.setField("KeyWords.keyword", "kw");
        r.save(conn);
        r = ih.getResource(conn, "Actor", "Organisaatio", "Name1", "NAME1");
        assertTrue(r.getKeywords().iterator().next().equals("kw"));
        r = ih.getResource(conn, "Actor", "Organisaatio", "Name2", "NAME1");
        assertTrue(r.getKeywords().iterator().next().equals("kw"));
        r = ih.getResource(conn, "Actor", "Organisaatio", "Name1", "NAME2");
        assertTrue(!r.getKeywords().iterator().hasNext());
        r = ih.getResource(conn, "Actor", "Joku", "Name1", "NAME1");
        assertTrue(!r.getKeywords().iterator().hasNext());

        r = ih.getResource(conn, "Channel", "Lehti", "Name2", "NAME2");
        r.setField("KeyWords.keyword", "kw");
        r.save(conn);
        r = ih.getResource(conn, "Channel", "Lehti", "Name2", "NAME2");
        assertTrue(r.getKeywords().iterator().next().equals("kw"));

        r = ih.getResource(conn, "Role", "Avustaja", "Name3", "NAME3");
        r.setField("KeyWords.keyword", "kw");
        r.save(conn);
        r = ih.getResource(conn, "Role", "Avustaja", "Name3", "NAME3");
        assertTrue(r.getKeywords().iterator().next().equals("kw"));

        r = ih.getResource(conn, "Document", null, "Name4", "NAME4");
        r.setField("KeyWords.keyword", "kw");
        r.save(conn);
        r = ih.getResource(conn, "Document", null, "Name4", "NAME4");
        assertTrue(r.getKeywords().iterator().next().equals("kw"));
    }
    
    public void testCreateRelation() throws Exception {
        resetDB();
        
        ih.createRelation(conn, null, "bogus", "bogus", false);
        ih.createRelation(conn, "bogus", null, "bogus", false);
        ih.createRelation(conn, "bogus", "bogus", null, false);
        
        Resource dr1 = new Resource("dr1", "Document");
        dr1.save(conn);
        Resource dr2 = new Resource("dr2", "Document");
        dr2.save(conn);
        Resource cr1 = new Resource("cr1", "Channel");
        cr1.save(conn);
        
        ih.createRelation(conn, "dr1", "dr2", "BIBREF", false);
        ih.createRelation(conn, "dr1", "dr2", "BIBREF", false);
        ih.createRelation(conn, "dr1", "dr2", "BIBREF", true);
        ih.createRelation(conn, "cr1", "dr1", "NONE", false);
        assertTrue(numRelations("dr1", "dr2", "BIBREF") == 1);
        assertTrue(numRelations("dr2", "dr1", "BIBREF") == 1);
        assertTrue(numRelations("cr1", "dr1", "NONE") == 1);
        assertTrue(tableIsEmpty("ResourceRelation"));
        
        Resource ar1 = new Resource("ar1", "Actor");
        ar1.setField("Actor.type", "Organisaatio");
        ar1.save(conn);
        Resource ar2 = new Resource("ar2", "Actor");
        ar2.setField("Actor.type", "Joku");
        ar2.save(conn);
        
        ih.createRelation(conn, "ar1", "dr1", "Julkaisija", false);
        ih.createRelation(conn, "ar2", "dr1", "Julkaisija", false);
        ih.createRelation(conn, "ar1", "dr1", "Julkaisija", false);
        ih.createRelation(conn, "ar2", "dr2", "Julkaisija", false);
        ih.createRelation(conn, "ar2", "dr2", "Hallinnoija", false);
        assertTrue(numRelations("ar1", "dr1", "Julkaisija") == 1);
        assertTrue(numRelations("ar2", "dr1", "Julkaisija") == 1);
        assertTrue(numRelations("ar2", "dr2", "Julkaisija") == 1);
        assertTrue(numRelations("ar2", "dr2", "Hallinnoija") == 1);
        assertTrue(tableIsEmpty("ResourceRelation"));
        assertTrue(numRoles("Julkaisija") == 2);
        assertTrue(numRoles("Hallinnoija") == 1);
        assertTrue(roleTableIsEmpty());
    }
    
    public void testIntegrateDocument() throws Exception {
        resetDB();
        
        Record rec = new Record("id");
        rec.setField("Nimeke", "Doku1 Hakusana");
        rec.setField("KanoNimeke", "DOKU1 HAKUSANA");
        rec.setField("Otsikko", "Dokaaminen");
        ih.integrateDocument(conn, "src1", rec);
        
        String resID = IntegratorHelper.createDocID(rec);
        
        Resource res = new Resource(resID, "Document");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Doku1 Hakusana"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("DOKU1 HAKUSANA"));
        assertTrue(res.getSingleField("Document.title").equals("Dokaaminen"));
        assertTrue(res.getSingleField("Records.src1¤id").equals(DbTools.formatDate(new Date())));
        Iterator i = res.getKeywords().iterator();
        assertTrue(i.next().equals("DOKU1"));
        assertTrue(i.next().equals("HAKUSANA"));
        assertTrue(!i.hasNext());
        
        rec.setField("Aihe", "Dokaaminen");
        rec.setField("HenkilöTekijä", "Kreivi Dooku");
        rec.setField("JokuKanava", "Saimaan kanava");
        ih.integrateDocument(conn, "src2", rec);
        
        res = new Resource(resID, "Document");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Doku1 Hakusana"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("DOKU1 HAKUSANA"));
        assertTrue(res.getSingleField("Document.title").equals("Dokaaminen"));
        assertTrue(res.getSingleField("Records.src1¤id").equals(DbTools.formatDate(new Date())));
        assertTrue(res.getSingleField("Records.src2¤id").equals(DbTools.formatDate(new Date())));
        i = res.getKeywords().iterator();
        assertTrue(i.next().equals("DOKU1"));
        assertTrue(i.next().equals("HAKUSANA"));
        assertTrue(i.next().equals("Dokaaminen"));
        assertTrue(!i.hasNext());
        
        res = new Resource("Kreivi Dooku", "Actor");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Kreivi Dooku"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("KREIVI DOOKU"));
        assertTrue(res.getSingleField("Actor.type").equals("Henkilö"));

        res = new Resource("Saimaan kanava", "Channel");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Saimaan kanava"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("SAIMAAN KANAVA"));
        assertTrue(res.getSingleField("Channel.type").equals("Joku"));

        assertTrue(numRelations("Kreivi Dooku", resID, "Tekijä") == 1);
        assertTrue(numRelations("Saimaan kanava", resID, "NONE") == 1);
        
        rec = new Record("id2");
        rec.setField("Nimeke", "Doku2");
        rec.setField("KanoNimeke", "DOKU2");
        rec.setField("HenkilöTekijä", "Kreivi Dooku");
        rec.setField("LehtiKanava", "Saimaan kanava");
        ih.integrateDocument(conn, "src1", rec);

        resID = IntegratorHelper.createDocID(rec);
        
        assertTrue(numRelations("Kreivi Dooku", resID, "Tekijä") == 1);
        assertTrue(numRelations("Saimaan kanava2", resID, "NONE") == 1);
        assertTrue(tableIsEmpty("ResourceRelation"));
        
        assertTrue(numActors("Kreivi Dooku") == 1);
        assertTrue(tableIsEmpty("Actor"));
        assertTrue(numChannels("Saimaan kanava") == 2);
        assertTrue(tableIsEmpty("Channel"));
    }

    public void testIntegrateNameQuickFormat() throws Exception {
        resetDB();
        
        Record rec;
        Resource res;
        Iterator i;

        rec = new Record("id");
        rec.setField("OrganisaatioToimija", "Helsingin Yliopisto");
        rec.setField("Aihe", "yliopisto");
        rec.setField("SisäinenTunniste", "HY");
        rec.setField("OrganisaatioHallinnoija", "Valtio");
        ih.integrateNameQuickFormat(conn, "src1", rec);
        
        res = new Resource("Helsingin Yliopisto", "Actor");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Helsingin Yliopisto"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("HELSINGIN YLIOPISTO"));
        assertTrue(res.getSingleField("Records.src1¤id").equals(DbTools.formatDate(new Date())));
        assertTrue(res.getSingleField("Actor.type").equals("Organisaatio"));
        i = res.getKeywords().iterator();
        assertTrue(i.next().equals("yliopisto"));
        assertTrue(!i.hasNext());
        i = res.getExtIDs().iterator();
        assertTrue(i.next().equals("HY"));
        assertTrue(!i.hasNext());
        
        res.setOverwrite(true);
        res.setField("Resource.name", "H:gin Yliopisto");
        res.save(conn);

        rec = new Record("id");
        rec.setField("OrganisaatioToimija", "Helsingin Yliopisto");
        rec.setField("Aihe", "Helsinki");
        rec.setField("SisäinenTunniste", "koulu");
        rec.setField("OrganisaatioHallinnoija", "Valtio");
        ih.integrateNameQuickFormat(conn, "src2", rec);

        res = new Resource("Helsingin Yliopisto", "Actor");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Helsingin Yliopisto"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("HELSINGIN YLIOPISTO"));
        assertTrue(res.getSingleField("Records.src1¤id").equals(DbTools.formatDate(new Date())));
        assertTrue(res.getSingleField("Records.src2¤id").equals(DbTools.formatDate(new Date())));
        assertTrue(res.getSingleField("Actor.type").equals("Organisaatio"));
        i = res.getKeywords().iterator();
        assertTrue(i.next().equals("Helsinki"));
        assertTrue(i.next().equals("yliopisto"));
        assertTrue(!i.hasNext());
        i = res.getExtIDs().iterator();
        assertTrue(i.next().equals("HY"));
        assertTrue(i.next().equals("koulu"));
        assertTrue(!i.hasNext());
        
        res = new Resource("Valtio", "Actor");
        res.load(conn);
        assertTrue(res.getSingleField("Resource.name").equals("Valtio"));
        assertTrue(res.getSingleField("Resource.canonicalName").equals("VALTIO"));
        assertTrue(res.getSingleField("Records.src1¤id").equals(DbTools.formatDate(new Date())));
        assertTrue(res.getSingleField("Records.src2¤id").equals(DbTools.formatDate(new Date())));
        assertTrue(res.getSingleField("Actor.type").equals("Organisaatio"));
        
        assertTrue(numRelations("Valtio", "Helsingin Yliopisto", "Hallinnoija") == 1);
        assertTrue(tableIsEmpty("ResourceRelation"));
        assertTrue(numActors("Helsingin Yliopisto") == 1);
        assertTrue(numActors("Valtio") == 1);
        assertTrue(tableIsEmpty("Actor"));
        assertTrue(numRoles("Hallinnoija") == 1);
        assertTrue(roleTableIsEmpty());
    }

    public void testResolveDocumentReferences() throws Exception {
        resetDB();
        
        Record rec1 = new Record("id1");
        rec1.setField("Nimeke", "Doku1");
        rec1.setField("KanoNimeke", "DOKU1");
        rec1.setField("Viitattu", "d2");
        rec1.setField("Viitattu", "d3");
        rec1.setField("Viittaava", "d2");
        rec1.setField("SisäinenTunniste", "d1");
        String r1ID = IntegratorHelper.createDocID(rec1);
        ih.integrateDocument(conn, "src", rec1);

        Record rec2 = new Record("id2");
        rec2.setField("Nimeke", "Doku2");
        rec2.setField("KanoNimeke", "DOKU2");
        rec2.setField("Viitattu", "d1");
        rec2.setField("Viittaava", "d1");
        rec2.setField("Viittaava", "d3");
        rec2.setField("SisäinenTunniste", "d2");
        String r2ID = IntegratorHelper.createDocID(rec2);
        ih.integrateDocument(conn, "src", rec2);
        
        Vector recs = new Vector();
        recs.add(rec1);
        recs.add(rec2);
        Iterator i = recs.iterator();
        while (i.hasNext()) {
            ih.resolveDocumentReferences(conn, (Record)i.next(), "src");
        }
//        ih.resolveDocumentReferences(conn, recs, "src");
        
        assertTrue(numRelations(r1ID, r2ID, "BIBREF") == 1);
        assertTrue(numRelations(r1ID, "d3", "BIBREF") == 1);
        assertTrue(numRelations(r2ID, r1ID, "BIBREF") == 1);
        assertTrue(numRelations("d3", r2ID, "BIBREF") == 1);
        assertTrue(tableIsEmpty("ResourceRelation"));
        
        Resource res = new Resource("d3", "Document");
        res.load(conn);
        assertTrue(res.getSingleField("Records.src¤id1").equals(DbTools.formatDate(new Date())));
        i = res.getExtIDs().iterator();
        assertTrue(i.next().equals("d3"));
        assertTrue(!i.hasNext());
    }
}
