/*
 * IntegratorImplTest.java
 *
 * This software is released under the GNU GPL license
 *
 */

package maito.integration;

import java.sql.*;
import java.util.HashMap;
import java.util.Properties;

import maito.util.Tools;
import maito.resource.Resource;
import maito.datacollecting.DataSourceDescription;

import junit.framework.TestCase;

public class IntegratorImplTest extends TestCase {
    private Connection conn = null;
    private Statement stmt = null;
    private IntegratorImpl ii = null;

	public IntegratorImplTest(String name) throws Exception {
		super(name);
		Class.forName("com.mysql.jdbc.Driver").newInstance();
		this.ii = new IntegratorImpl();
	}
	
	public void setUp() throws Exception {
		conn = DriverManager.getConnection(
	        "jdbc:mysql://localhost/maitotest?user=root&password=pass");
		assertTrue(conn != null);
		stmt = conn.createStatement();
    }
    
    public void tearDown() throws Exception {
        stmt = null;
        conn.close();
    }

    private void resetDB() throws Exception {        
        stmt.execute("TRUNCATE TABLE resourcerelation;");
        stmt.execute("TRUNCATE TABLE actor;");
        stmt.execute("TRUNCATE TABLE channel;");
        stmt.execute("TRUNCATE TABLE document;");
        stmt.execute("TRUNCATE TABLE extids;");
        stmt.execute("TRUNCATE TABLE keywords;");
        stmt.execute("TRUNCATE TABLE records;");
        stmt.execute("DELETE FROM role WHERE NOT id='NONE' AND NOT id='BIBREF';");
        stmt.execute("DELETE FROM resource WHERE NOT id='NONE' AND NOT id='BIBREF';");
    }

    /*
     * Actual test methods begin
     */
    
    public void testAddGraph() throws Exception {
        Properties dbConfig = Tools.loadProperties(Tools.PATH_DBCONFIG);
        assertTrue(dbConfig != null);

        try {
            ii.addGraph(null);
            assertTrue(false);
        } catch(IllegalArgumentException e) {}
        try {
            ii.addGraph("");
            assertTrue(false);
        } catch(IllegalArgumentException e) {}
        
        ii.addGraph("unittest");
        
        ResultSet rs = stmt.executeQuery("SHOW DATABASES");
        boolean found = false;
        while (rs.next()) {
            if (rs.getString(1).equals(dbConfig.getProperty("dbname_resources")+"_unittest")) {
                found = true;
                break;
            }
        }
        rs.close();
        assertTrue(found);
        
        HashMap hm = new HashMap();
        stmt.execute("USE "+dbConfig.getProperty("dbname_resources")+"_unittest");
        rs = stmt.executeQuery("SHOW TABLES");
        while (rs.next()) {
            hm.put(rs.getString(1), "");
        }
        rs.close();
        stmt.execute("DROP DATABASE "+dbConfig.getProperty("dbname_resources")+"_unittest");
        assertTrue(hm.get("resource") != null);
        assertTrue(hm.get("extids") != null);
        assertTrue(hm.get("keywords") != null);
        assertTrue(hm.get("records") != null);
        assertTrue(hm.get("datasources") != null);
        assertTrue(hm.get("actor") != null);
        assertTrue(hm.get("role") != null);
        assertTrue(hm.get("document") != null);
        assertTrue(hm.get("channel") != null);
        assertTrue(hm.get("resourcerelation") != null);
    }
    
    public void testIntegrate() throws Exception {
        Properties dbConfig = Tools.loadProperties(Tools.PATH_DBCONFIG);

        stmt.execute("USE "+dbConfig.getProperty("dbname_rawdata"));
        ResultSet rs = stmt.executeQuery("SELECT COUNT(id) FROM DataSource WHERE id IN('__src1','__scr2')");
        rs.next();
        assertTrue(rs.getInt(1) == 0);
        
        stmt.execute("INSERT INTO DataSource SET id='__src1',format='dcxml',location=''");
        stmt.execute("INSERT INTO DataRecord SET id='d1',source='__src1'");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src1','d1','Nimeke','Dokari1')");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src1','d1','KanoNimeke','DOKARI1')");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src1','d1','URITunniste','d1')");
        stmt.execute("INSERT INTO DataRecord SET id='d2',source='__src1'");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src1','d2','Nimeke','Dokari2')");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src1','d2','KanoNimeke','DOKARI2')");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src1','d2','Viitattu','d1')");
        stmt.execute("INSERT INTO DataSource SET id='__src2',format='quick_format_name',location=''");
        stmt.execute("INSERT INTO DataRecord SET id='pekka',source='__src2'");
        stmt.execute("INSERT INTO Statement (source,record,name,value) VALUES ('__src2','pekka','JokuJulkaisija','Pekka')");
    
        ii.addGraph("unittest");
        stmt.execute("USE "+dbConfig.getProperty("dbname_resources")+"_unittest");
        resetDB();
    
        DataSourceDescription[] ds = new DataSourceDescription[2];
        ds[0] = new DataSourceDescription("__src1", "", "", 1, "", false);
        ds[1] = new DataSourceDescription("__src2", "", "", 1, "quick_format_name", false);
        
        ii.integrate(ds, "unittest");
        while (ii.workInProgress()) {
            Thread.sleep(1000);
        }
        
        Resource r;
        r = new Resource("xxxXXDokari1", "Document");
        assertTrue(r.load(conn));
        assertTrue(r.getSingleField("Resource.name").equals("Dokari1"));
        assertTrue(r.getSingleField("Resource.canonicalName").equals("DOKARI1"));
        
        r = new Resource("xxxXXDokari2", "Document");
        assertTrue(r.load(conn));
        assertTrue(r.getSingleField("Resource.name").equals("Dokari2"));
        assertTrue(r.getSingleField("Resource.canonicalName").equals("DOKARI2"));
        
        r = new Resource("Pekka", "Role");
        assertTrue(r.load(conn));
        assertTrue(r.getSingleField("Resource.name").equals("Pekka"));
        assertTrue(r.getSingleField("Resource.canonicalName").equals("PEKKA"));
        
        assertTrue(stmt.executeQuery("SELECT role FROM ResourceRelation WHERE subject='xxxXXDokari2' AND object='xxxXXDokari1' AND role='BIBREF'").next());
        
        stmt.execute("DROP DATABASE "+dbConfig.getProperty("dbname_resources")+"_unittest");
        stmt.execute("USE "+dbConfig.getProperty("dbname_rawdata"));
        
        rs = stmt.executeQuery("SELECT integratedTo FROM DataRecord WHERE source IN('__src1','__src2')");
        assertTrue(rs.next());
        assertTrue(rs.getString(1).equals("unittest,"));
        assertTrue(rs.next());
        assertTrue(rs.getString(1).equals("unittest,"));
        rs.close();
        
        stmt.execute("DELETE FROM DataSource WHERE id IN('__src1','__src2')");
    }
}
