/*
 * ResourceTest.java
 *
 * This software is released under the GNU GPL license
 *
 */

package maito.resource;

import java.sql.*;

import maito.resource.Resource;

import junit.framework.TestCase;

public class ResourceTest extends TestCase {
    private Connection conn = null;
    private Statement stmt = null;

	public ResourceTest(String name) throws Exception {
		super(name);
		Class.forName("com.mysql.jdbc.Driver").newInstance();
	}
	
	public void setUp() throws Exception {
		conn = DriverManager.getConnection(
	        "jdbc:mysql://localhost/maitotest?user=maito&password=maito");
		assertTrue(conn != null);
		stmt = conn.createStatement();
    }
    
    public void tearDown() throws Exception {
        stmt = null;
        conn.close();
    }

    private void resetDB() throws Exception {        
        stmt.execute("TRUNCATE TABLE actor;");
        stmt.execute("TRUNCATE TABLE channel;");
        stmt.execute("TRUNCATE TABLE document;");
        stmt.execute("TRUNCATE TABLE extids;");
        stmt.execute("TRUNCATE TABLE keywords;");
        stmt.execute("TRUNCATE TABLE records;");
        stmt.execute("TRUNCATE TABLE resource;");
        stmt.execute("TRUNCATE TABLE role;");
    }
        

    public void testOverwrite() throws Exception {
        Resource r = new Resource("res1", "Document");
        assertTrue(!r.getOverwrite());
        r.setOverwrite(true);
        assertTrue(r.getOverwrite());
    }

    public void testFields() throws Exception {
        Resource r = new Resource("res1", "Document");
        
        assertTrue(!r.setField(null, null));
        assertTrue(!r.setField("Bogus.name", null));
        assertTrue(!r.setField("Resource.bogus", null));
        assertTrue(!r.setField("Role.bogus", null));
        assertTrue(!r.setField("Actor.bogus", null));
        assertTrue(!r.setField("Channel.bogus", null));
        assertTrue(!r.setField("Document.bogus", null));
        assertTrue(!r.setField("KeyWords.bogus", null));
        assertTrue(!r.setField("ExtIDs.bogus", null));
        assertTrue(!r.setField("", null));
        assertTrue(!r.setField(".", null));
        assertTrue(!r.setField("Resource.", null));
//        assertTrue(!r.setField("Records.", null));

        assertTrue(r.getKeywords().isEmpty());
        assertTrue(r.getExtIDs().isEmpty());

        r.setOverwrite(false);
        
        assertTrue(r.setField("Resource.name", null));
        assertTrue(r.setField("Resource.canonicalName", ""));
        assertTrue(r.setField("Role.type", null));
        assertTrue(r.setField("Document.published", null));
        assertTrue(r.setField("Document.language", ""));
        assertTrue(r.setField("Document.title", "dokari"));
        assertTrue(r.setField("KeyWords.keyword", "keyword1"));
        assertTrue(r.setField("ExtIDs.extID", "extID1"));
        assertTrue(r.setField("Records.", "blah"));
        assertTrue(r.setField("Records.blah", null));

        assertTrue(r.getSingleField(null) == null);
        assertTrue(r.getSingleField("") == null);
        assertTrue(r.getSingleField(".") == null);
        assertTrue(r.getSingleField("Resource.") == null);
        assertTrue(r.getSingleField("Bogus.name") == null);
        assertTrue(r.getSingleField("Resource.bogus") == null);
        assertTrue(r.getSingleField("Role.bogus") == null);
        assertTrue(r.getSingleField("Actor.bogus") == null);
        assertTrue(r.getSingleField("Channel.bogus") == null);
        assertTrue(r.getSingleField("Document.bogus") == null);
        assertTrue(r.getSingleField("Records.bogus") == null);
        
        assertTrue(r.getSingleField("Resource.name") == null);
        assertTrue(r.getSingleField("Resource.canonicalName").equals(""));
        assertTrue(r.getSingleField("Role.type") == null);
        assertTrue(r.getSingleField("Document.published") == null);
        assertTrue(r.getSingleField("Document.language").equals(""));
        assertTrue(r.getSingleField("Document.title").equals("dokari"));
        assertTrue(r.getSingleField("Records.").equals("blah"));
        assertTrue(r.getSingleField("Records.blah") == null);
        assertTrue(r.getKeywords().size() == 1);
        assertTrue(r.getExtIDs().size() == 1);

        assertTrue(r.setField("Resource.name", "nimi"));
        assertTrue(r.setField("Resource.canonicalName", "kanonimi"));
        assertTrue(r.setField("Role.type", ""));
        assertTrue(r.setField("Document.published", "1955"));
        assertTrue(r.setField("Document.language", "fi"));
        assertTrue(!r.setField("Document.title", "doku"));
        assertTrue(r.setField("KeyWords.keyword", "keyword1"));
        assertTrue(r.setField("ExtIDs.extID", "extID1"));
        assertTrue(!r.setField("Records.", "blah2"));
        assertTrue(r.setField("Records.blah", "blah2"));

        assertTrue(r.getSingleField("Resource.name").equals("nimi"));
        assertTrue(r.getSingleField("Resource.canonicalName").equals("kanonimi"));
        assertTrue(r.getSingleField("Role.type").equals(""));
        assertTrue(r.getSingleField("Document.published").equals("1955"));
        assertTrue(r.getSingleField("Document.language").equals("fi"));
        assertTrue(r.getSingleField("Document.title").equals("dokari"));
        assertTrue(r.getSingleField("Records.").equals("blah"));
        assertTrue(r.getSingleField("Records.blah").equals("blah2"));
        assertTrue(r.getKeywords().size() == 1);
        assertTrue(r.getExtIDs().size() == 1);

        assertTrue(!r.setField("Resource.name", "nimi2"));
        assertTrue(r.setField("Role.type", "guest star"));
        assertTrue(r.setField("KeyWords.keyword", "keyword2"));
        assertTrue(r.setField("ExtIDs.extID", "extID2"));

        assertTrue(r.getSingleField("Resource.name").equals("nimi"));
        assertTrue(r.getSingleField("Role.type").equals("guest star"));
        assertTrue(r.getKeywords().size() == 2);
        assertTrue(r.getExtIDs().size() == 2);

        assertTrue(!r.setField("Role.type", "evening star"));
        assertTrue(r.setField("KeyWords.keyword", "keyword2"));
        assertTrue(r.setField("ExtIDs.extID", "extID2"));

        assertTrue(r.getSingleField("Role.type").equals("guest star"));
        assertTrue(r.getKeywords().size() == 2);
        assertTrue(r.getKeywords().contains("keyword1"));
        assertTrue(r.getKeywords().contains("keyword2"));
        assertTrue(r.getExtIDs().size() == 2);
        assertTrue(r.getExtIDs().contains("extID1"));
        assertTrue(r.getExtIDs().contains("extID2"));

        r.setOverwrite(true);
        
        assertTrue(r.setField("Resource.name", "overwritten"));
        assertTrue(r.setField("Role.type", "overwritten"));
        assertTrue(r.setField("Document.published", "overwritten"));
        assertTrue(r.setField("Records.blah", "overwritten"));

        assertTrue(r.getSingleField("Resource.name").equals("overwritten"));
        assertTrue(r.getSingleField("Role.type").equals("overwritten"));
        assertTrue(r.getSingleField("Document.published").equals("overwritten"));
        assertTrue(r.getSingleField("Records.blah").equals("overwritten"));
    }
    
    public void testLoadAndSave() throws Exception {
        resetDB();
        
        Resource r = new Resource("res1", "Document");
        assertTrue(!r.load(conn));

        assertTrue(r.setField("Resource.name", "nimi"));
        assertTrue(r.setField("Resource.canonicalName", "kanonimi"));
        assertTrue(r.setField("Document.referenceInfo", "refinfo"));
        assertTrue(r.setField("Document.published", "2006-01-20"));
        assertTrue(r.setField("Document.language", "fi"));
        assertTrue(r.setField("Document.title", "Dokari"));
        assertTrue(r.setField("KeyWords.keyword", "keyword1"));
        assertTrue(r.setField("KeyWords.keyword", "keyword2"));
        assertTrue(r.setField("ExtIDs.extID", "extID1"));
        assertTrue(r.setField("ExtIDs.extID", "extID2"));
        assertTrue(r.setField("Records.source¤record", "2005-01-01"));
        assertTrue(r.save(conn));
        
        r = new Resource("res1", "Document");
//        assertTrue(!r.load(null));
        assertTrue(r.load(conn));
        assertTrue(r.getSingleField("Resource.name").equals("nimi"));
        assertTrue(r.getSingleField("Resource.canonicalName").equals("kanonimi"));
        assertTrue(r.getSingleField("Document.referenceInfo").equals("refinfo"));
        assertTrue(r.getSingleField("Document.published").equals("2006-01-20"));
        assertTrue(r.getSingleField("Document.language").equals("fi"));
        assertTrue(r.getSingleField("Document.title").equals("Dokari"));
        assertTrue(r.getSingleField("Records.source¤record").equals("2005-01-01"));
        assertTrue(r.getKeywords().size() == 2);
        assertTrue(r.getKeywords().contains("keyword1"));
        assertTrue(r.getKeywords().contains("keyword2"));
        assertTrue(r.getExtIDs().size() == 2);
        assertTrue(r.getExtIDs().contains("extID1"));
        assertTrue(r.getExtIDs().contains("extID2"));
        
        r.setOverwrite(true);
        assertTrue(r.setField("Resource.name", "Uusi nimi"));
        assertTrue(r.setField("KeyWords.keyword", "uusi kw"));
        assertTrue(r.setField("ExtIDs.extID", "uusi extid"));
        assertTrue(r.setField("Records.sorsa¤rekkari", "2001-01-01"));
        assertTrue(r.save(conn));

        r = new Resource("res1", "Document");
        assertTrue(r.load(conn));
        assertTrue(r.getSingleField("Resource.name").equals("Uusi nimi"));
        assertTrue(r.getSingleField("Records.source¤record").equals("2005-01-01"));
        assertTrue(r.getSingleField("Records.sorsa¤rekkari").equals("2001-01-01"));
        assertTrue(r.getKeywords().size() == 3);
        assertTrue(r.getKeywords().contains("keyword1"));
        assertTrue(r.getKeywords().contains("keyword2"));
        assertTrue(r.getKeywords().contains("uusi kw"));
        assertTrue(r.getExtIDs().size() == 3);
        assertTrue(r.getExtIDs().contains("extID1"));
        assertTrue(r.getExtIDs().contains("extID2"));
        assertTrue(r.getExtIDs().contains("uusi extid"));
    }
    
    public void testFindUniqueID() throws Exception {
        resetDB();
    
        Resource r = new Resource("res1", "Document");
        assertTrue(r.setField("Resource.name", "Hauki"));
        assertTrue(r.setField("Resource.canonicalName", "HAUKI"));
//        assertTrue(r.findUniqueID(null) == null);
        assertTrue(r.findUniqueID(conn).equals("res1"));
        assertTrue(r.save(conn));
        assertTrue(r.findUniqueID(conn).equals("res12"));
	}
}
