/**
* ToolsTest.java
* 
* v1
*
* 10.11.2005
* 
* This software is released under the GNU GPL license
* 
* @author Vin Ala-Hrknen
* 
* Note: coverage at 99% since the private dummy constructor
* of Tools can't be called.
*/
package maito.util;

import junit.framework.TestCase;

public class ToolsTest extends TestCase {

    /*
     * Test method for 'maito.util.Tools.canonizeGeneric(String)'
     */
    public void testCanonizeGeneric() {
        assertTrue(Tools.canonizeGeneric(null) == null);
        assertTrue(Tools.canonizeGeneric(" \n\t kk\n  \t vsytt  \t ").equals("KK VSYTT"));
        assertTrue(Tools.canonizeGeneric("http://dUmMy*()**fOo???.com/~foOBAr/").equals("HTTP://DUMMYFOO.COM/~FOOBAR/"));
        assertTrue(Tools.canonizeGeneric("abcdefghijklmnopqrstuvwxyz,./-:~*&%#!").equals("ABCDEFGHIJKLMNOPQRSTUVWXYZ,./-:~"));
        assertTrue(Tools.canonizeGeneric("YOU KNOW, THIS IS ALREADY CANONIZED.").equals("YOU KNOW, THIS IS ALREADY CANONIZED."));
    }

    /*
     * Test method for 'maito.util.Tools.canonizeActor(String, int)'
     */
    public void testCanonizeActor() {
        
        // with a null value
        String[] results;
        assertTrue(Tools.canonizeActor(null, Tools.ACTOR_CITESEER) == null);
        
        // The special exception rule
        results = Tools.canonizeActor("http://foobar.com/", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Joku") && results[1].equals("http://foobar.com/"));
        
        // Organization keyword recognition + canonization
        results = Tools.canonizeActor("Microsoft Corp.", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Organisaatio") && results[1].equals("MICROSOFT CORP."));
        results = Tools.canonizeActor("mark of the unicorn", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Organisaatio") && results[1].equals("MARK OF THE UNICORN"));
        
        // Actor recognition with person keywords
        results = Tools.canonizeActor("Dijkstra et al.", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("DIJKSTRA"));
        results = Tools.canonizeActor("Mini-me jr.", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("MINI-ME"));
        
        // Actor recognition with surname initial
        results = Tools.canonizeActor("Malcolm M.", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil"));
        results = Tools.canonizeActor("Malcolm M. Jones", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil"));
        
        // Default value when nothing else detected
        results = Tools.canonizeActor("i'm not an actor, really", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Joku") && results[1].equals("IM NOT AN ACTOR, REALLY"));
        
         // Citeseer format person name canonization
        results = Tools.canonizeActor("Malcolm Jr.", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("MALCOLM"));
        results = Tools.canonizeActor("Malcolm K. Jones", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("JONES, M"));
        results = Tools.canonizeActor("Malcolm K. Jones,Geoff  McDowell ", Tools.ACTOR_CITESEER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("JONES, M") && results[2].equals("MCDOWELL, G"));
        
        // "Other" format person name canonization
        results = Tools.canonizeActor("Malcolm Jr.", Tools.ACTOR_OTHER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("MALCOLM"));
        results = Tools.canonizeActor("Jones, Malcolm K.", Tools.ACTOR_OTHER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("JONES, M"));
        results = Tools.canonizeActor("Jones Malcolm K.", Tools.ACTOR_OTHER);
        assertTrue(results[0].equals("Henkil") && results[1].equals("JONES, M"));
    }

    /*
     * Test method for 'maito.util.Tools.canonizeDate(String)'
     */
    public void testCanonizeDate() {
        assertTrue(Tools.canonizeDate(null) == null);
        assertTrue(Tools.canonizeDate("1983-2-1") == null);
        assertTrue(Tools.canonizeDate("1983-11-kk") == null);
        assertTrue(Tools.canonizeDate("1983-11-07").equals("1983-11-07"));
        assertTrue(Tools.canonizeDate("1997-07-16T19:20+01:00").equals("1997-07-16"));
        assertTrue(Tools.canonizeDate("1997-07-16therecanbeanycrapinhere").equals("1997-07-16"));
    }

    /*
     * Test method for 'maito.util.Tools.canonizeLang(String)'
     */
    public void testCanonizeLang() {
        assertTrue(Tools.canonizeLang(null) == null);
        assertTrue(Tools.canonizeLang("test") == null);
        assertTrue(Tools.canonizeLang("t") == null);
        assertTrue(Tools.canonizeLang("123") == null);
        assertTrue(Tools.canonizeLang("123-tes") == null);
        assertTrue(Tools.canonizeLang("test-uk") == null);
        assertTrue(Tools.canonizeLang("fi").equals("FI"));
        assertTrue(Tools.canonizeLang("fin").equals("FIN"));
        assertTrue(Tools.canonizeLang("en-uk").equals("EN"));
        assertTrue(Tools.canonizeLang("en-12345)?#%!?%(!").equals("EN"));
        assertTrue(Tools.canonizeLang("fin-suomi").equals("FIN"));
    }

    /*
     * Test method for 'maito.util.Tools.canonizeTitle(String)'
     */
    public void testCanonizeTitle() {
        assertTrue(Tools.canonizeTitle(null) == null);
        assertTrue(Tools.canonizeTitle("the importance of testing in software design projects pt. 2").equals("IMPORTANCE TESTING SOFTWARE DESIGN PROJECTS"));
    }

    /*
     * Test method for 'maito.util.Tools.categorizeIdentifier(String)'
     */
    public void testCategorizeIdentifier() {
        assertTrue(Tools.categorizeIdentifier(null) == null);
        
        // Standard heuristics
        assertTrue(Tools.categorizeIdentifier("http://www.fi./").equals("URLTunniste"));
        assertTrue(Tools.categorizeIdentifier("urn:foo:a123,456").equals("URNTunniste"));
        assertTrue(Tools.categorizeIdentifier("INFO:OAI/arXiv.org:hep-th%2F9901001").equals("OAITunniste"));
        assertTrue(Tools.categorizeIdentifier("info:ofi/fmt:kev:mtx:sch_svc").equals("OpenURLTunniste"));
        assertTrue(Tools.categorizeIdentifier("info:doi/10.1000.10/123456 ").equals("DOITunniste"));
        assertTrue(Tools.categorizeIdentifier("info:foobar").equals("INFOTunniste"));
        assertTrue(Tools.categorizeIdentifier("jotain://foobar").equals("URITunniste"));
        assertTrue(Tools.categorizeIdentifier("foobar").equals("SisinenTunniste"));
        
        // Reference ID heuristics
        assertTrue(Tools.categorizeIdentifier("1983...something.").equals("ViittausTunniste"));
        assertTrue(Tools.categorizeIdentifier("something....1983").equals("ViittausTunniste"));
        assertTrue(Tools.categorizeIdentifier("somethingidentifier1983").equals("SisinenTunniste"));
        assertTrue(Tools.categorizeIdentifier("somet.hi,ngidentifier2019").equals("ViittausTunniste"));
        assertTrue(Tools.categorizeIdentifier("somet.hi,ngidentifier1799").equals("SisinenTunniste"));
        assertTrue(Tools.categorizeIdentifier("somet.hi,ngidentifier2020").equals("SisinenTunniste"));
    }

}
