/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <string.h>

#include <lemon/concepts/graph.h>
#include <lemon/list_graph.h>
#include <lemon/euler.h>

#include "io.h"
#include "align.h"

#define BUF_LEN 1024
#define THRESHOLD 1000

void read_scaffolds(ifstream *lp, long genome_length,
                    int *num_scaffolds, set<int> **scaffolds,
                    map<int,long> **contig2pos, int *contig_length,
                    char **contig_seq, int max_overlap, align_t *align,
                    double max_align_err) {
  lemon::ListGraph G;
  lemon::ListGraph::NodeMap<int> node2contig(G);
  map<int, lemon::ListGraph::Node> contig2node;
  (*contig2pos) = new map<int,long>;

  char buffer[BUF_LEN];

  // look for x[integer]whitespace[float]
  while(!lp->getline(buffer, BUF_LEN).eof()) {
    if (buffer[0] == '*') {
      break;
    }
    if (buffer[0] == 'x') {
      char *p, *p2;
      int contig, pos;

      p = &buffer[1];
      p2 = p;
      while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
      *p2 = '\0';

      contig = atoi(p);

      p = p2+1;
      while(*p == ' ' || *p == '\t') p++;
      p2 = p;
      while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
      *p2 = '\0';
      
      pos = (long)atof(p);

      if (pos >= 0) {
        pos = pos + genome_length/2;
      } else {
        pos = pos - genome_length/2;
      }

      // Check for overlaps
      int overlap = 0;
      for(lemon::ListGraph::NodeIt n2(G); n2 != lemon::INVALID; ++n2) {
        long s1, s2, e1, e2;
        int c1, c2;

        c1 = contig;
        if (pos >= 0) {
          s1 = pos;
          e1 = s1 + contig_length[c1];
        } else {
          e1 = -pos;
          s1 = e1 - contig_length[c1];
        }

        c2 = node2contig[n2];
        if ((*(*contig2pos))[c2] >= 0) {
          s2 = (*(*contig2pos))[c2];
          e2 = s2 + contig_length[c2];
        } else {
          e2 = -(*(*contig2pos))[c2];
          s2 = e2 - contig_length[c2];
        }

        long ol = 0;
        if (s1 < s2) {
          if (e1 < e2) {
            ol = e1-s2;
          } else {
            ol = e2-s2;
          }
        } else {
          if (e1 < e2) {
            ol = e1-s1;
          } else {
            ol = e2-s1;
          }
        }
      
        if (ol > max_overlap) {
          if (s1 <= s2 && e2 <= e1) {
            if (is_contained(contig_seq[c2], (*(*contig2pos))[c2] > 0,
                             contig_length[c2], contig_seq[c1], 
                             pos > 0, contig_length[c1], 
                             s2-s1-1000, s2-s1+1000, align, max_align_err)) {
              printf("Contig %d is contained in contig %d\n", c2, c1);
              ol = 0;
            } 
          } else if (s2 <= s1 && e1 <= e2) {
            if (is_contained(contig_seq[c1], pos > 0,
                             contig_length[c1], contig_seq[c2], 
                             (*(*contig2pos))[c2] > 0, contig_length[c2], 
                             s1-s2-1000, s1-s2+1000, align, max_align_err)) {
              printf("Contig %d is contained in contig %d\n", c1, c2);
              ol = 0;
            } 
          } else if (s1 <= s2) {
            if (is_overlap(contig_seq[c1], pos > 0,
                           contig_length[c1], contig_seq[c2], 
                           (*(*contig2pos))[c2] > 0, contig_length[c2],
                           s2-s1-1000, s2-s1+1000, align, max_align_err)) {
              printf("Contig %d overlaps with contig %d\n", c1, c2);
              ol = 0;
            }
          } else {
            if (is_overlap(contig_seq[c2], (*(*contig2pos))[c2] > 0,
                           contig_length[c2], contig_seq[c1], 
                           pos > 0, contig_length[c1],
                           s1-s2-1000, s1-s2+1000, align, max_align_err)) {
              printf("Contig %d overlaps with contig %d\n", c2, c1);
              ol = 0;
            }
          }

          if (ol) {
            overlap = 1;
            break;
          }
        }
      }

      if (!overlap) {
        lemon::ListGraph::Node n = G.addNode();
        contig2node[contig] = n;
        node2contig[n] = contig;
        (*(*contig2pos))[contig] = pos;
      } else {
        contig2node[contig] = lemon::INVALID;
      }
    }
#ifdef ORI_VAR

    if (buffer[0] == 'o') {
      char *p, *p2;
      int contig;
      float ori;

      p = &buffer[1];
      p2 = p;
      while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
      *p2 = '\0';

      contig = atoi(p);

      p = p2+1;
      while(*p == ' ' || *p == '\t') p++;
      p2 = p;
      while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
      *p2 = '\0';
      
      ori = atof(p);
      
      if (ori < 0.5) {
        (*(*contig2pos))[contig] = 
          -((*(*contig2pos))[contig] + contig_length[contig]);
      }
    }
#endif

    if (buffer[0] == 'I') {
      char *p, *p2;
      int contig1, contig2;
      double val;

      p = &buffer[1];
      p2 = p;
      while(*p2 != '_' && *p2 != '\0') p2++;
      *p2 = '\0';

      contig1 = atoi(p);

      p = p2+1;
      p2 = p;
      while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
      *p2 = '\0';

      contig2 = atoi(p);

      p = p2+1;
      while(*p == ' ' || *p == '\t') p++;
      p2 = p;
      while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
      *p2 = '\0';
      
      val = atof(p);

      if (contig2node[contig1] != lemon::INVALID && 
          contig2node[contig2] != lemon::INVALID &&
          (1.0-val)*2.0*(double)genome_length < THRESHOLD) {
        /* --> -->  */
        if ((contig1 & 0x01) && !(contig2 & 0x01)) {
          if (((*(*contig2pos))[contig1/2] > 0 &&
               (*(*contig2pos))[contig2/2] > 0) ||
              ((*(*contig2pos))[contig1/2] < 0 &&
               (*(*contig2pos))[contig2/2] < 0)) {

            G.addEdge(contig2node[contig1/2], contig2node[contig2/2]);
          }
        }

        /* --> <--  */
        if ((contig1 & 0x01) && (contig2 & 0x01)) {
          if (((*(*contig2pos))[contig1/2] > 0 &&
               (*(*contig2pos))[contig2/2] < 0) ||
              ((*(*contig2pos))[contig1/2] < 0 &&
               (*(*contig2pos))[contig2/2] > 0)) {
            G.addEdge(contig2node[contig1/2], contig2node[contig2/2]);
          }
        }

        /* <-- -->  */
        if (!(contig1 & 0x01) && !(contig2 & 0x01)) {
          if (((*(*contig2pos))[contig1/2] < 0 &&
               (*(*contig2pos))[contig2/2] > 0) ||
              ((*(*contig2pos))[contig1/2] > 0 &&
               (*(*contig2pos))[contig2/2] < 0)) {
            G.addEdge(contig2node[contig1/2], contig2node[contig2/2]);
          }
        }

        /* <-- <--  */
        if (!(contig1 & 0x01) && (contig2 & 0x01)) {
          if (((*(*contig2pos))[contig1/2] > 0 &&
               (*(*contig2pos))[contig2/2] > 0) ||
              ((*(*contig2pos))[contig1/2] < 0 &&
               (*(*contig2pos))[contig2/2] < 0)) {
        
            G.addEdge(contig2node[contig1/2], contig2node[contig2/2]);
          }
        }
      }
    }
  }

  // Component analysis
  lemon::ListGraph::NodeMap<int> components(G);
  int num;
  
  num = connectedComponents(G, components);

  *scaffolds = new set<int>[num];

  for(lemon::ListGraph::NodeIt n(G); n != lemon::INVALID; ++n) {
    (*scaffolds)[components[n]].insert(node2contig[n]);
  }

  *num_scaffolds = num;
}
