/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <string.h>
#include <stdlib.h>
#include <limits.h>

#include "cio.h"

using namespace std;

#define BUF_LEN 1024

int read_contigs(char *contig_file, int **contig_length, char ***contigs,
                 char ***contig_name, map<string,int> *name2id) {
  ifstream stream;
  long int len = -1;
  int id = -1;
  char buffer[BUF_LEN];
  int count = 0;

  char *data;

  printf("Reading contigs from %s\n", contig_file);

  // Count contigs
  stream.open(contig_file);

  if (!stream.is_open()) {
    printf("Could not open contig file\n");
    exit(2);
  }

  len = 0;
  while (!(stream).getline(buffer, BUF_LEN).eof()) {
    if (stream.fail())
      stream.clear();
    if (buffer[0] == '>') {
      count++;
    }
    len += strlen(buffer)+1;
  }
  stream.close();

  printf("Reserving space: %ld\n", len);

  *contig_length = new int[count];
  *contigs = (char **) malloc(count*sizeof(char *));
  *contig_name = (char **) malloc(count*sizeof(char *));
  data = new char[len];

  stream.open(contig_file);
  len = 0;
  id = 0;
  while (!(stream).getline(buffer, BUF_LEN).eof()) {
    if (stream.fail())
      stream.clear();
    if (buffer[0] == '>') {
      if (id != 0) {
          data[len] = '\0';
          len++;
      }
      for(long int i = 0; i < (signed long int)strlen(buffer); i++)
        if (buffer[i] == ' ' || buffer[i] == '\n')
          buffer[i] = '\0';
      strcpy(&data[len], &buffer[1]);
      (*contig_name)[id] = &data[len];
      (*name2id)[(*contig_name)[id]] = id;
      len += strlen(&buffer[1])+1;
      (*contigs)[id] = &data[len];
      id++;
    } else {
      strcpy(&data[len], buffer);
      len += strlen(buffer);
    }
  }
  data[len] = '\0';
  len++;
  stream.close();

  for(id = 0; id < count; id++) {
    (*contig_length)[id] = strlen((*contigs)[id]);
  }

  return count;
}

void read_contig_depth_info(char * depth_file, map<string,int> *name2id, 
                            char **contig_name, double *contig_depth) {
  ifstream depth;
  char buf[BUF_LEN];
  char *p, *p2;
  int contig;
  char *name;
  int i = 0;

  depth.open(depth_file);
  while(!(depth.getline(buf, BUF_LEN)).eof()) {
    if (depth.fail()) {
      printf("Too long line in coverage file\n");
      abort();
    }
    p = p2 = buf;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;

    p2++;
    while(*p2 == ' ' || *p2 == '\t') p2++;
      
    // contig name
    name = p2;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
    *p2 = '\0';
    p2++;
    while(*p2 == ' ' || *p2 == '\t') p2++;
      
    // contig length
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') p2++;
    while(*p2 == ' ' || *p2 == '\t') p2++;
      
    p = p2;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0' && *p2 != '\n') p2++;
    *p2 = '\0';

    contig = (*name2id)[name];

    if (strcmp(contig_name[contig], name) == 0) {
        contig_depth[contig] = atof(p);
        i++;
    }

  }
  depth.close();

  printf("Read depth info for %d contigs\n", i);
}


int read_config_file(char *config_file, 
                     char **filename, int *insert_size, int *min, int *max) {
  ifstream conf;

  char buf[BUF_LEN];
  char *p, *p2;
  int i;

  conf.open(config_file);

  if (!conf.is_open()) {
    printf("Could not open config file\n");
    exit(2);
  }

  i = 0;
  while(!(conf.getline(buf, BUF_LEN)).eof()) {
    if (conf.fail()) {
      printf("Too long line in configuration file\n");
      abort();
    }
    p = p2 = buf;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') {
      p2++;
    }
    if (*p2 == '\0')
      break;
    *p2 = '\0';

    p2++;
    while(*p2 == ' ' || *p2 == '\t') *p2++;
    
    p = p2;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') {
      p2++;
    }
    if (*p2 == '\0')
      break;
    *p2 = '\0';
    
    filename[i] = (char *)malloc((strlen(p)+1)*sizeof(char));
    strcpy(filename[i], p);

    p2++;
    while(*p2 == ' ' || *p2 == '\t') *p2++;
    
    p = p2;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') {
      p2++;
    }
    if (*p2 == '\0')
      break;
    *p2 = '\0';
    
    insert_size[i] = atoi(p);

    p2++;
    while(*p2 == ' ' || *p2 == '\t') *p2++;
    
    p = p2;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') {
      p2++;
    }
    if (*p2 == '\0')
      break;
    *p2 = '\0';
    min[i] = atoi(p);

    p2++;
    while(*p2 == ' ' || *p2 == '\t') *p2++;
    
    p = p2;
    while(*p2 != ' ' && *p2 != '\t' && *p2 != '\0') {
      p2++;
    }
    *p2 = '\0';

    max[i] = atoi(p);

    i++;
  }

  conf.close();

  return i;
}
