/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include "blocks.h"
#include <iostream>
#include <stdio.h>

using namespace std;

blocks::node *blocks::make_vertex(long label) {
  struct node *newnode = new node;
  newnode->block_id = blocks.maketree(0, label);
  newnode->component_id = components.makeset(label);
  return newnode;
}

bool blocks::insert_edge(struct node *u, struct node *v, long label, long max_block_size) {
  long uc, vc;
  long bl;

  uc = components.find(u->component_id);
  vc = components.find(v->component_id);

  if (uc != vc) {
    bl = blocks.maketree(label, 0);

    if (components.getrank(uc) < components.getrank(vc)) {
      blocks.evert(u->block_id);
      blocks.link(u->block_id, bl);
      blocks.link(bl, v->block_id);
    } else {
      blocks.evert(v->block_id);
      blocks.link(v->block_id, bl);
      blocks.link(bl, u->block_id);
    }
    components.Union(uc, vc);

    return 1;
  } else {
    list<long> l = blocks.find_path(u->block_id, v->block_id);
    int size = 1;

    for(list<long>::iterator it = l.begin(); it != l.end(); it++) {
      if (blocks.get_circle_label(*it) != 0)
        size += blocks.getnode_count(*it)-1;
    }

    if  (size <= max_block_size) {
      blocks.condense_path(l, label);
      return 1;
    } else {
      return 0;
    }
  }

  return 0;
}

bool blocks::insert_edge2(struct node *u, struct node *v, long label, long max_block_edges) {
  long uc, vc;
  long bl;

  uc = components.find(u->component_id);
  vc = components.find(v->component_id);

  if (uc != vc) {
    bl = blocks.maketree(label, 0);

    if (components.getrank(uc) < components.getrank(vc)) {
      blocks.evert(u->block_id);
      blocks.link(u->block_id, bl);
      blocks.link(bl, v->block_id);
    } else {
      blocks.evert(v->block_id);
      blocks.link(v->block_id, bl);
      blocks.link(bl, u->block_id);
    }
    components.Union(uc, vc);

    return 1;
  } else {
    list<long> l = blocks.find_path(u->block_id, v->block_id);
    int size = 0;

    for(list<long>::iterator it = l.begin(); it != l.end(); it++) {
      if (blocks.get_circle_label(*it) != 0) {
        size += blocks.getedge_count(*it);
      }
    }

    if  (size < max_block_edges) {
      blocks.condense_path(l, label);
      return 1;
    } else {
      return 0;
    }
  }

  return 0;
}


void blocks::enumerate_blocks(map<long,list<long> * > *block2nodes, list<struct node *> nodes) {
  list<long> *block;

  for(list<struct node *>::iterator it = nodes.begin(); it != nodes.end(); it++) {
    struct node *n = *it;
    long bl, bl_id;

    bl = blocks.getparent(n->block_id);
    if (bl != n->block_id) {
      bl_id = blocks.get_circle_label(bl);
      block = (*block2nodes)[bl_id];
      if (block == NULL) {
        block = new list<long>;
        (*block2nodes)[bl_id] = block;
        long up = blocks.getparent(bl);
        if (up != bl) {
          block->push_back(blocks.get_square_label(up));
        }
      }
      block->push_back(blocks.get_square_label(n->block_id));
    }
  }
}


#ifdef TEST
int main () { 
  blocks::node *s1, *s2, *s3, *s4, *s5, *s6, *s7, *s8, *s9, *s0;
  blocks b;

  list<blocks::node * > nodes;
  map<long, list<long> * > block2nodes;

  s1 = b.make_vertex(1);
  s2 = b.make_vertex(2);
  s3 = b.make_vertex(3);
  s4 = b.make_vertex(4);
  s5 = b.make_vertex(5);
  s6 = b.make_vertex(6);
  s7 = b.make_vertex(7);
  s8 = b.make_vertex(8);
  s9 = b.make_vertex(9);
  s0 = b.make_vertex(10);

  nodes.push_back(s1);
  nodes.push_back(s2);
  nodes.push_back(s3);
  nodes.push_back(s4);
  nodes.push_back(s5);
  nodes.push_back(s6);
  nodes.push_back(s7);
  nodes.push_back(s8);
  nodes.push_back(s9);
  nodes.push_back(s0);


  if (b.insert_edge2(s1, s2, 1, 10)) {
    cout << "1 --- 2\n";
  }
  if (b.insert_edge2(s2, s3, 2, 10)) {
    cout << "2 --- 3\n";
  }
  if (b.insert_edge2(s1, s3, 3, 10)) {
    cout << "1 --- 3\n";
  }
  if (b.insert_edge2(s3, s4, 4, 10)) {
    cout << "3 --- 4\n";
  }
  if (b.insert_edge2(s5, s6, 5, 10)) {
    cout << "5 --- 6\n";
  }
  if (b.insert_edge2(s4, s6, 6, 10)) {
    cout << "4 --- 6\n";
  }
  if (b.insert_edge2(s4, s5, 7, 10)) {
    cout << "4 --- 5\n";
  }
  if (b.insert_edge2(s2, s5, 8, 10)) {
    cout << "2 --- 5\n";
  }

  if (b.insert_edge2(s7, s8, 9, 10)) {
    cout << "7 --- 8\n";
  }
  if (b.insert_edge2(s8, s9, 10, 10)) {
    cout << "8 --- 9\n";
  }
  if (b.insert_edge2(s9, s0, 11, 10)) {
    cout << "9 --- 0\n";
  }

  if (b.insert_edge2(s0, s7, 12, 10)) {
    cout << "0 --- 7\n";
  }

  if (b.insert_edge2(s8, s4, 13, 10)) {
    cout << "8 --- 4\n";
  }

  if (b.insert_edge2(s0, s8, 14, 10)) {
    cout << "0 --- 8\n";
  }

  b.enumerate_blocks(&block2nodes, nodes);
  
  for(map<long, list<long> *>::iterator it = block2nodes.begin(); it != block2nodes.end(); it++) {
    cout << "Block " << it->first << "\n";

    for(list<long>::iterator lit = (*it->second).begin(); lit != (*it->second).end(); lit++) {
      cout << "   " << *lit << "\n";
    }
  }

  return 0;
}
#endif
