/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef CONDENSIBLE_NODE_H
#define CONDENSIBLE_NODE_H
#include <stdlib.h>
#include <iostream>
#include <list>
#include <map>
using namespace std;

#include "disjoint-set.h"

template<class TC, class TS> class CondensibleNodeForest {
 private:
    struct node {
        long parent;
        TC circle_label;
        TS square_label;
        long self;
        long node_count;
        long edge_count;
    };

    DisjointSetForest<long> subnodes;

 public:
    CondensibleNodeForest() {}
    long maketree(TC circle_label, TS square_label);
    void evert(long root);
    void link(long x, long y);
    list<long> find_path(long u, long v);
    void condense_path(list<long> path, TC circle_label);
    TC get_circle_label(long a);
    TS get_square_label(long a);
    long getnode_count(long a);
    long getedge_count(long a);
    long getparent(long a);
};

template<class TC, class TS> TC CondensibleNodeForest<TC, TS>::get_circle_label(long a) {
    node *n = (node *) a;
    return n->circle_label;
}

template<class TC, class TS> TS CondensibleNodeForest<TC, TS>::get_square_label(long a) {
    node *n = (node *) a;
    return n->square_label;
}

template<class TC, class TS> long CondensibleNodeForest<TC, TS>::getnode_count(long a) {
    node *n = (node *) a;

    // Check if this is a root
    if (subnodes.getvalue(subnodes.find(n->parent)) == a) {
        return n->node_count;
    } else {
        return n->node_count+1;
    }
}

template<class TC, class TS> long CondensibleNodeForest<TC, TS>::getedge_count(long a) {
    node *n = (node *) a;

    return n->edge_count;
}

template<class TC, class TS> long CondensibleNodeForest<TC, TS>::getparent(long a) {
    node *n = (node *) a;

    return subnodes.getvalue(subnodes.find(n->parent));
}

template<class TC, class TS> long CondensibleNodeForest<TC, TS>::maketree(TC circle_label, TS square_label) {
  node *newnode = new node;

  newnode->self = subnodes.makeset((long)newnode);
  newnode->circle_label = circle_label;
  newnode->square_label = square_label;
  newnode->parent = newnode->self;
  newnode->node_count = 0;
  newnode->edge_count = 0;

  return (long)newnode;
}

template<class TC, class TS> void CondensibleNodeForest<TC, TS>::evert(long root) {
  node *current = (node *)root;
  node *next;
  node *parent;

  current->node_count++;

  next = (node *)subnodes.getvalue(subnodes.find(current->parent));
  current->parent = current->self;
  while(next != current) {
      if ((long)next == root) {
          cout << "Evert loop\n";
      }
      parent = (node *)subnodes.getvalue(subnodes.find(next->parent));
      next->parent = current->self;
      current = next;
      next = parent;
  }

  current->node_count--;

  current = (node *)root;
}

/* Make x (root!) child of y */
template<class TC, class TS> void CondensibleNodeForest<TC, TS>::link(long x, long y) {
    node *xnode = (node *)x;
    node *ynode = (node *)y;

    if (subnodes.getvalue(subnodes.find(xnode->parent)) != x) {
        cout << "Invalid argument to link\n";
        abort();
    }

    xnode->parent = ynode->self;
    ynode->node_count++;
    ynode->edge_count++;
}

template<class TC, class TS> list<long> CondensibleNodeForest<TC, TS>::find_path(long u, long v) {
    list<long> ul, vl;
    map<node *, bool> visited;
    node *cu = (node *)u;
    node *cv = (node *)v;
    node *temp;

    if (cu->circle_label != 0 || cv->circle_label != 0) {
        cout << "Invalid argument to find_path\n";
        abort();
    }

    visited[cu] = 1;
    visited[cv] = 1;

    if (cu == cv) {
        ul.push_back(subnodes.find(cu->parent));
        return ul;
    }

    while(1) {
        temp = (node *)subnodes.getvalue(subnodes.find(cu->parent));
        if (temp != cu) {
            cu = temp;
            if (!visited[cu]) {
                visited[cu] = 1;
                ul.push_back((long)cu);
            } else {
                /* cu is the lowest common ancestor! */
                while(vl.size() > 0 && vl.back() != (long)cu) {
                    vl.pop_back();
                }
                ul.insert(ul.end(), vl.begin(), vl.end());
                return ul;
            }
        }

        temp = (node *)subnodes.getvalue(subnodes.find(cv->parent));
        if (temp != cv) {
            cv = temp;
            if (!visited[cv]) {
                visited[cv] = 1;
                vl.push_back((long)cv);
            } else {
                /* cv is the lowest common ancestor! */
                while(ul.size() > 0 && ul.back() != (long)cv) {
                    ul.pop_back();
                }
                vl.insert(vl.end(), ul.begin(), ul.end());
                return vl;
            }
        }
    }

    return ul;
}

template<class TC, class TS> void CondensibleNodeForest<TC, TS>::condense_path(list<long> path, TC label) {
    node *lca = (node *)path.back();
    node *merge = NULL;
    node *top;

    path.pop_back();

    if (lca->circle_label != 0) {
        // circle lca
        merge = lca;
    }

    for(list<long>::iterator it = path.begin(); it != path.end(); it++) {
        node *current = (node *)*it;
        if (current->circle_label == 0) {
            // square node
        } else {
            // circle node
            if (merge == NULL) {
                merge = current;
                node *n = (node *)subnodes.getvalue(subnodes.find(current->parent));
                if (n != merge)
                    n->node_count--;
                merge->parent = lca->self;
                lca->node_count++;
            } else {
                subnodes.Union(current->self, merge->self);
                merge->node_count += current->node_count;
                merge->edge_count += current->edge_count;
                node *n = (node *)subnodes.getvalue(subnodes.find(current->parent));
                if (n != current)
                    n->node_count--;
                current->node_count = merge->node_count;
                current->edge_count = merge->edge_count;
                current->parent = merge->parent;
            }
        }
    }

    merge->circle_label = label;
    merge->edge_count++;

    top = (node *)subnodes.getvalue(subnodes.find(merge->self));
    top->circle_label = merge->circle_label;
    top->node_count = merge->node_count;
    top->edge_count = merge->edge_count;
    top->parent = merge->parent;
}

#endif
