/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "graph.h"
#include "blocks.h"
#include "align.h"

#include <stack>

#include <omp.h>

lemon::ListGraph G;
map<int,lemon::ListGraph::Node> nodes;
lemon::ListGraph::NodeMap<int> node2id(G);
lemon::ListGraph::NodeMap<int> node2contig(G);
lemon::ListGraph::EdgeMap<int> support(G);
lemon::ListGraph::EdgeMap<int> avg_dist(G);
lemon::ListGraph::EdgeMap<int> min_dist(G);
lemon::ListGraph::EdgeMap<int> max_dist(G);
lemon::ListGraph::EdgeMap<int> min_span1(G);
lemon::ListGraph::EdgeMap<int> max_span1(G);
lemon::ListGraph::EdgeMap<int> min_span2(G);
lemon::ListGraph::EdgeMap<int> max_span2(G);

lemon::ListGraph::Edge get_edge(lemon::ListGraph::Node node1,
                                lemon::ListGraph::Node node2) {
  for(lemon::ListGraph::IncEdgeIt e(G, node1); e != lemon::INVALID; ++e) {
    if (G.u(e) == node2 || G.v(e) == node2) {
      return e;
    }
  }

  return lemon::INVALID;
}

int degree_histogram(int **degrees) {
  int max = 0;
  int count;

  int *d;

  for(lemon::ListGraph::NodeIt n(G); n != lemon::INVALID; ++n) {
    count = 0;
    for(lemon::ListGraph::IncEdgeIt e(G, n); e != lemon::INVALID; ++e) {
      count++;
    }
    if (count > max)
      max = count;
  }

  d = new int[max+1];

  for(int i = 0; i < max+1; i++) {
    d[i] = 0;
  }


  for(lemon::ListGraph::NodeIt n(G); n != lemon::INVALID; ++n) {
    count = 0;
    for(lemon::ListGraph::IncEdgeIt e(G, n); e != lemon::INVALID; ++e) {
      count++;
    }
    d[count]++;
  }

  *degrees = d;
  return max;
}

void fix_impossible_edges() {
  int count = 0;
  int avg;

  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    if (min_dist[e] > max_dist[e]) {
      count++;
      avg = (min_dist[e] + max_dist[e])/2;
      min_dist[e] = avg-100;
      max_dist[e] = avg+100;
    }
  }

  printf("Fixed %d impossible edges\n", count);

}

#define MAX_OVERLAP 500

void remove_negative_edges(char **contig, int *contig_len, int min_support,
                           int check, double max_align_err) {
  int removed = 0;
  int count = 0;
  align_t *align;

  int done;
  lemon::ListGraph::EdgeIt e(G);
  lemon::ListGraph::Edge mye;

  lemon::ListGraph::EdgeMap<int> to_remove(G);

  omp_set_num_threads(8);

#pragma omp parallel private(align, mye, done)
  {
    align = NULL;
    align = init_align();
    done = 0;
    while(1) {

#pragma omp critical
      {
        if (e == lemon::INVALID) {
          done = 1;
        } else {
          mye = e;
          ++e;
          to_remove[mye] = 0;
        }
      }

      if (done)
        break;

      if (support[e] >= min_support) {
        int dist;
#pragma omp critical 
        {
#ifdef AVG_DIST
          dist = avg_dist[mye];
#else
          dist = (max_dist[mye]+min_dist[mye])/2;
#endif
        }
        int remove = 0;
        if (dist < -MAX_OVERLAP) {
          int id1, id2;
          
#pragma omp critical
          {
            id1 = node2id[G.u(mye)];
            id2 = node2id[G.v(mye)];

            count++;
            if (count%100000 == 0)
              printf("%d/%d negative edges removed so far...\n", removed, 
                     count);
          }
          if (check) {
            if (contig_len[id1/2] < -dist) {
              if (!is_contained(contig[id1/2], id1 & 0x01, contig_len[id1/2],
                                contig[id2/2], !(id2 & 0x01), contig_len[id2/2],
                                contig_len[id2/2] + dist - 1000, 
                                contig_len[id2/2] + dist + 1000, align, max_align_err)) {
                remove = 1;
              }
            } else if (contig_len[id2/2] < -dist) {
              if (!is_contained(contig[id2/2], id2 & 0x01, contig_len[id2/2],
                                contig[id1/2], !(id1 & 0x01), contig_len[id1/2],
                                contig_len[id1/2] + dist - 1000, 
                                contig_len[id1/2] + dist + 1000, align, max_align_err)) {
                remove = 1;
              }
            } else {
              if (!is_overlap(contig[id1/2], id1 & 0x01, contig_len[id1/2],
                              contig[id2/2], !(id2 & 0x01), contig_len[id2/2],
                              contig_len[id1/2] + dist - 1000, 
                              contig_len[id1/2] + dist + 1000, align, max_align_err)) {
                remove = 1;
              }
            }
          } else {
            remove = 1;
          }
        }

        if (remove) {
#pragma omp critical
          {
            removed++;
            to_remove[mye] = 1;
          }
        }
      }
    }
  }

  for(lemon::ListGraph::EdgeIt e2(G); e2 != lemon::INVALID;) {
    lemon::ListGraph::Edge e3;
    if (to_remove[e2]) {
      e3 = e2;
      ++e2;
      G.erase(e3);
    } else {
      ++e2;
    }
  }

  printf("Removed %d/%d negative edges\n", removed, count);

}


void disconnect_high_degree_nodes(int degree_limit) {
  int count = 0;

  for(lemon::ListGraph::NodeIt n(G); n != lemon::INVALID;++n) {
    int degree = 0;

    for(lemon::ListGraph::IncEdgeIt e(G, n); e != lemon::INVALID; ++e) {
      if (max_dist[e] >= 0)
        degree++;
    }

    if (degree > degree_limit) {
      count++;
      for(lemon::ListGraph::IncEdgeIt e(G, n); e != lemon::INVALID;) {
        lemon::ListGraph::Edge e2 = e;
        ++e;
        G.erase(e2);
      }
    }
  }

  printf("Disconnected %d contig ends with too high degree\n", count);

}

lemon::ListGraph::Edge *edges;

int cmp_support(const void *p1, const void *p2) {
  lemon::ListGraph::Edge *e1, *e2;

  e1 = (lemon::ListGraph::Edge *)p1;
  e2 = (lemon::ListGraph::Edge *)p2;

  return support[*e2] - support[*e1];
}

void shuffle(lemon::ListGraph::Edge *edges, int size) {
  int i, j;
  lemon::ListGraph::Edge swap;

  srand(time(NULL));

  for(i = 0; i < size; i++) {
    j = i + rand() % (size-i);
    swap = edges[i];
    edges[i] = edges[j];
    edges[j] = swap;
  }
}

map<long,list<long> * > *remove_low_support_links(double *contig_depth, int *contig_length, int min_support, int max_block_size, double depth_limit, int length_limit) {
  int count;
  int i, j;
  lemon::ListGraph::NodeMap<blocks::node *> node2blocknode(G);
  blocks bl;
  list<blocks::node * > bnodes;
  map<long, list<long> * > *block2nodes = new map<long, list<long> *>;
  int remove_low, remove_comp;

  count = 0;
  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    count++;
  }

  edges = new lemon::ListGraph::Edge[count];
  i = 0;
  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    edges[i] = e;
    i++;
  }
  
  // sort the edges according to support
  qsort(edges, count, sizeof(lemon::ListGraph::Edge), cmp_support);
  //shuffle(edges, count);

  for(lemon::ListGraph::NodeIt n(G); n != lemon::INVALID; ++n) {
    node2blocknode[n] = NULL;
  }

  

  for(lemon::ListGraph::NodeIt n(G); n != lemon::INVALID; ++n) {
    int contig = node2contig[n];

    if (contig_depth[contig] <= depth_limit &&
        contig_length[contig] >= length_limit) {
      if ((node2id[n] & 0x01) == 0) {
        if (node2blocknode[nodes[node2id[n]+1]] == NULL) {
          node2blocknode[n] = bl.make_vertex(node2id[n]/2);
          bnodes.push_back(node2blocknode[n]);
        } else {
          node2blocknode[n] = node2blocknode[nodes[node2id[n]+1]];
        }
      } else {
        if (node2blocknode[nodes[node2id[n]-1]] == NULL) {
          node2blocknode[n] = bl.make_vertex(node2id[n]/2);
          bnodes.push_back(node2blocknode[n]);
        } else {
          node2blocknode[n] = node2blocknode[nodes[node2id[n]-1]];
        }
      }
    }
  }

  j = 0;
  remove_low = 0;
  remove_comp = 0;
  for(i = 0; i < count; i++) {
    lemon::ListGraph::Node u,v;
    int id1, id2;

    if (support[edges[i]] >= min_support) {

      u = G.u(edges[i]);
      v = G.v(edges[i]);

      id1 = node2contig[u];
      id2 = node2contig[v];

      if (contig_depth[id1] <= depth_limit && 
          contig_depth[id2] <= depth_limit &&
          contig_length[id1] >= length_limit &&
          contig_length[id2] >= length_limit) {
        
#if 0
        if (!bl.insert_edge(node2blocknode[u], node2blocknode[v], i+1, max_block_size)) {
          G.erase(edges[i]);
          j++;
          remove_comp++;
        }
#else
        if (!bl.insert_edge2(node2blocknode[u], node2blocknode[v], i+1, max_block_size)) {
          G.erase(edges[i]);
          j++;
          remove_comp++;
        }
#endif
      }
    } else {
      G.erase(edges[i]);
      j++;
      remove_low++;
    }
  }

  printf("Removed %d edges with too low support\n", remove_low);
  printf("Removed %d edges to keep biconnected components small\n", remove_comp);

  bl.enumerate_blocks(block2nodes, bnodes);

  return block2nodes;
}
