/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <string.h>
#include <stdlib.h>
#include <limits.h>

#include <lemon/concepts/graph.h>
#include <lemon/list_graph.h>

#include <list>

#include "io.h"
#include "graph.h"

#include "lp_lib.h"

using namespace std;

#define BUF_LEN 4096

char *tok;

char *tokenize(char *str, const char delim) {
  int i;

  if (str == NULL)
    str = tok;

  for(i = 0; i < (signed int) strlen(str); i++) {
    if (str[i] == delim) {
      str[i] = '\0';
      tok = &str[i+1];
      break;
    }
  }

  return str;
}

int read_mapping_line(ifstream *mappings, 
                      char *contig1, int *start1, int *end1, char *ori1, 
                      char *contig2, int *start2, int *end2, char *ori2) {
  char buffer[BUF_LEN];
  char *field;
  int i;

  if ((*mappings).getline(buffer, BUF_LEN).eof()) {
    /* No more mapping lines left */
    return 1;
  }

  if ((*mappings).fail()) {
    printf("Too long lines in mapping file\n");
    abort();
  }

  /* Tokenize the string */
  /* mate pair name */
  field = tokenize(buffer, '\t');
  /* Orientation 1 */
  field = tokenize(NULL, '\t');
  i = atoi(field);
  if (i & 0x10) {
    *ori1 = 'R';
  } else {
    *ori1 = 'F';
  }
  /* contig name 1*/
  field = tokenize(NULL, '\t');
  for(i = 0; i < (signed int)strlen(field); i++) {
    contig1[i] = field[i];
  }
  contig1[i]= '\0';
  /* start pos */
  field = tokenize(NULL, '\t');
  *start1 = atoi(field);
  /* mapping quality */
  field = tokenize(NULL, '\t');
  /* CIGAR string -> figure out end pos */
  field = tokenize(NULL, '\t');
  *end1 = *start1;
  while(*field != '\0') {
    i = atoi(field);
    while(*field != '\0' && *field >= '0' && *field <= '9') field++;
    if (*field == 'M' || *field == 'I' || *field == 'S' || *field == '=' ||
        *field == 'X')
      *end1 += i;
    if (*field != '\0')
      field++;
  }

  /* skip five fields */
  field = tokenize(NULL, '\t');
  field = tokenize(NULL, '\t');
  field = tokenize(NULL, '\t');
  field = tokenize(NULL, '\t');
  field = tokenize(NULL, '\t');

  /* mate pair name */
  field = tokenize(NULL, '\t');
  /* Orientation 2*/
  field = tokenize(NULL, '\t');
  i = atoi(field);
  if (i & 0x10) {
    *ori2 = 'R';
  } else {
    *ori2 = 'F';
  }
  /* contig name 2*/
  field = tokenize(NULL, '\t');
  for(i = 0; i < (signed int)strlen(field); i++) {
    contig2[i] = field[i];
  }
  contig2[i]= '\0';
  /* start pos */
  field = tokenize(NULL, '\t');
  *start2 = atoi(field);
  /* mapping quality */
  field = tokenize(NULL, '\t');
  /* CIGAR string -> figure out end pos */
  field = tokenize(NULL, '\t');
  *end2 = *start2;
  while(*field != '\0') {
    i = atoi(field);
    while(*field != '\0' && *field >= '0' && *field <= '9') field++;
    if (*field == 'M' || *field == 'I' || *field == 'S' || *field == '=' ||
        *field == 'X')
      *end2 += i;
    if (*field != '\0')
      field++;
  }

  return 0;
}

lemon::ListGraph::EdgeMap<list<int> > mate_min(G);
lemon::ListGraph::EdgeMap<list<int> > mate_max(G);
lemon::ListGraph::EdgeMap<list<int> > mate_avg(G);

int compare_int(const void *p1, const void *p2) {
  int i1 = *((int *)p1);
  int i2 = *((int *)p2);

  return i1 - i2;
}

void construct_scaffold_graph(char **mapping_file, int num_contigs, 
                              int *contig_length, map<string, int>* name2id,
                              int *ins, int *min_ins, int *max_ins, int libs) {
  ifstream mappings;

  char contig1[BUF_LEN];
  char contig2[BUF_LEN];
  char ori1, ori2;
  int start1, start2, end1, end2;
  int id1, id2;

  int ecount = 0;

  int min, max;

  int insert, min_insert, max_insert;
  int k;

  for(k = 0; k < libs; k++) {
    insert = ins[k];
    min_insert = min_ins[k];
    max_insert = max_ins[k];

    printf("Reading mappings %s\n", mapping_file[k]);

    mappings.open(mapping_file[k]);

    if (!mappings.is_open()) {
      printf("Could not open mappings file\n");
      exit(2);
    }

    while(!read_mapping_line(&mappings, contig1, &start1, &end1, &ori1,
                             contig2, &start2, &end2, &ori2)) {

      // printf("1: %s %d %d %c\n", contig1, start1, end1, ori1);
      // printf("2: %s %d %d %c\n", contig2, start2, end2, ori2);

      id1 = (*name2id)[contig1];
      id2 = (*name2id)[contig2];
      if (id1 != id2) {
        lemon::ListGraph::Edge e = lemon::INVALID;
        if (ori1 == 'F' && ori2 == 'F') {
          // -------------------------->   -------------------------->
          //                   --->           --->
          //                   R3             F3
          e = get_edge(nodes[2*id1], nodes[2*id2+1]);
          if (e == lemon::INVALID) {
            e = G.addEdge(nodes[2*id1], nodes[2*id2+1]);
            support[e] = 1;

            avg_dist[e] = insert - (contig_length[id2] - start2) - end1;
            min_dist[e] = min_insert - (contig_length[id2] - start2) - end1;
            max_dist[e] = max_insert - (contig_length[id2] - start2) - end1;

            min_span1[e] = INT_MAX;
            max_span1[e] = 0;
            min_span2[e] = INT_MAX;
            max_span2[e] = 0;
            ecount++;
          } else {
            avg_dist[e] += insert - (contig_length[id2] - start2) - end1;
            if (min_dist[e] < min_insert - (contig_length[id2] - start2) - end1)
              min = min_insert - (contig_length[id2] - start2) - end1;
            else
              min = min_dist[e];
            if (max_dist[e] > max_insert - (contig_length[id2] - start2) - end1)
              max = max_insert - (contig_length[id2] - start2) - end1;
            else
              max = max_dist[e];

            support[e]++;
            if (max >= min) {
              if (min_dist[e] < min_insert - (contig_length[id2] - start2) - end1)
                min_dist[e] = min_insert - (contig_length[id2] - start2) - end1;
              if (max_dist[e] > max_insert - (contig_length[id2] - start2) - end1)
                max_dist[e] = max_insert - (contig_length[id2] - start2) - end1;
            } else {
              // e = lemon::INVALID;
            }
          }
          mate_min[e].push_back(min_insert - (contig_length[id2] - start2) - 
                                end1);
          mate_max[e].push_back(max_insert - (contig_length[id2] - start2) - 
                                end1);
          mate_avg[e].push_back(insert - (contig_length[id2] - start2) - 
                                end1);
        } else if (ori1 == 'F' && ori2 == 'R') {
          // <--------------------------   -------------------------->
          //                   --->           --->
          //                   R3             F3
          e = get_edge(nodes[2*id1], nodes[2*id2]);
          if (e == lemon::INVALID) {
            e = G.addEdge(nodes[2*id1], nodes[2*id2]);
            support[e] = 1;

            avg_dist[e] = insert - end1 - end2;
            min_dist[e] = min_insert - end1 - end2;
            max_dist[e] = max_insert - end1 - end2;

            min_span1[e] = INT_MAX;
            max_span1[e] = 0;
            min_span2[e] = INT_MAX;
            max_span2[e] = 0;
            ecount++;
          } else {
            avg_dist[e] += insert - end1 - end2;
            if (min_dist[e] < min_insert - end1 - end2) 
              min = min_insert - end1 - end2;
            else
              min = min_dist[e];
            if (max_dist[e] > max_insert - end1 - end2)
              max = max_insert - end1 - end2;
            else
              max = max_dist[e];
          
            support[e]++;
            if (max >= min) {
              if (min_dist[e] < min_insert - end1 - end2) 
                min_dist[e] = min_insert - end1 - end2;
              if (max_dist[e] > max_insert - end1 - end2)
                max_dist[e] = max_insert - end1 - end2;
            } else {
              // e = lemon::INVALID;
            }
          }
          mate_min[e].push_back(min_insert - end1 - end2);
          mate_max[e].push_back(max_insert - end1 - end2);
          mate_avg[e].push_back(insert - end1 - end2);
        } else if (ori1 == 'R' && ori2 == 'F') {
          // -------------------------->  <--------------------------
          //                   --->           --->
          //                   R3             F3
          e = get_edge(nodes[2*id1+1], nodes[2*id2+1]);
          if (e == lemon::INVALID) {
            e = G.addEdge(nodes[2*id1+1], nodes[2*id2+1]);
            support[e] = 1;

            avg_dist[e] = insert - (contig_length[id1] - start1) 
              - (contig_length[id2] - start2);
            min_dist[e] = min_insert - (contig_length[id1] - start1) 
              - (contig_length[id2] - start2);
            max_dist[e] = max_insert - (contig_length[id1] - start1) 
              - (contig_length[id2] - start2);

            min_span1[e] = INT_MAX;
            max_span1[e] = 0;
            min_span2[e] = INT_MAX;
            max_span2[e] = 0;
            ecount++;
          } else {
            avg_dist[e] += insert - (contig_length[id1] - start1) 
              - (contig_length[id2] - start2);
            if (min_dist[e] < min_insert - (contig_length[id1] - start1) 
                - (contig_length[id2] - start2))
              min = min_insert - (contig_length[id1] - start1) 
                - (contig_length[id2] - start2);
            else
              min = min_dist[e];
            if (max_dist[e] > max_insert - (contig_length[id1] - start1) 
                - (contig_length[id2] - start2))
              max = max_insert - (contig_length[id1] - start1) 
                - (contig_length[id2] - start2);
            else
              max = max_dist[e];

            support[e]++;
            if (max >= min) {
              if (min_dist[e] < min_insert - (contig_length[id1] - start1) 
                  - (contig_length[id2] - start2))
                min_dist[e] = min_insert - (contig_length[id1] - start1) 
                  - (contig_length[id2] - start2);
              if (max_dist[e] > max_insert - (contig_length[id1] - start1) 
                  - (contig_length[id2] - start2))
                max_dist[e] = max_insert - (contig_length[id1] - start1) 
                  - (contig_length[id2] - start2);
            } else {
              // e = lemon::INVALID;
            }
          }
          mate_min[e].push_back(min_insert - (contig_length[id1] - start1) 
                                - (contig_length[id2] - start2));
          mate_max[e].push_back(max_insert - (contig_length[id1] - start1) 
                                - (contig_length[id2] - start2));
          mate_avg[e].push_back(insert - (contig_length[id1] - start1) 
                                - (contig_length[id2] - start2));
        } else if (ori1 == 'R' && ori2 == 'R') {
          // <--------------------------   <--------------------------
          //                   --->           --->
          //                   R3             F3
          e = get_edge(nodes[2*id1+1], nodes[2*id2]);
          if (e == lemon::INVALID) {
            e = G.addEdge(nodes[2*id1+1], nodes[2*id2]);
            support[e] = 1;

            avg_dist[e] = insert - end2 - (contig_length[id1] - start1);
            min_dist[e] = min_insert - end2 - (contig_length[id1] - start1);
            max_dist[e] = max_insert - end2 - (contig_length[id1] - start1);

            min_span1[e] = INT_MAX;
            max_span1[e] = 0;
            min_span2[e] = INT_MAX;
            max_span2[e] = 0;
            ecount++;
          } else {
            avg_dist[e] += insert - end2 - (contig_length[id1] - start1);
            if (min_dist[e] < min_insert - end2 - (contig_length[id1] - start1))
              min = min_insert - end2 - (contig_length[id1] - start1);
            else
              min = min_dist[e];
            if (max_dist[e] > max_insert - end2 - (contig_length[id1] - start1))
              max = max_insert - end2 - (contig_length[id1] - start1);
            else
              max = max_dist[e];

            support[e]++;
            if (max >= min) {
              support[e]++;
              if (min_dist[e] < min_insert - end2 - (contig_length[id1] - start1))
                min_dist[e] = min_insert - end2 - (contig_length[id1] - start1);
              if (max_dist[e] > max_insert - end2 - (contig_length[id1] - start1))
                max_dist[e] = max_insert - end2 - (contig_length[id1] - start1);
            } else {
              // e = lemon::INVALID;
            }
          }
          mate_min[e].push_back(min_insert - end2 -(contig_length[id1] - start1));
          mate_max[e].push_back(max_insert - end2 -(contig_length[id1] - start1));
          mate_avg[e].push_back(insert - end2 -(contig_length[id1] - start1));
        }

        if (e != lemon::INVALID) {
          if (id1 < id2) {
            if (start1 < min_span1[e])
              min_span1[e] = start1;
            if (end1 > max_span1[e])
              max_span1[e] = end1;
            if (start2 < min_span2[e])
              min_span2[e] = start2;
            if (end2 > max_span2[e])
              max_span2[e] = end2;
          } else {
            if (start2 < min_span1[e])
              min_span1[e] = start2;
            if (end2 > max_span1[e])
              max_span1[e] = end2;
            if (start1 < min_span2[e])
              min_span2[e] = start1;
            if (end1 > max_span2[e])
              max_span2[e] = end1;
          }
        }
      }
    }

    mappings.close();
  }

#ifdef EDGE_BUNDLING
  int max_support = 0;
  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    if (support[e] > max_support)
      max_support = support[e];
  }
  // int sorted_min[max_support];
  // int sorted_max[max_support];
  int *sorted_min = new int[max_support];
  int *sorted_max = new int[max_support];
  int i, j, len;
  list<int>::iterator it1, it2, it3;

  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    i = 0;
    it1 = mate_min[e].begin(); it2 = mate_max[e].begin();
    while(it1 != mate_min[e].end() && it2 != mate_max[e].end()) {
      sorted_min[i] = (int) *it1;
      sorted_max[i] = (int) *it2;
      i++;
      it1++; it2++;
    }
    len = i;

    qsort(sorted_min, len, sizeof(int), compare_int);
    qsort(sorted_max, len, sizeof(int), compare_int);

    i = 0;
    j = 0;
    int open = 0;
    int max_open = 0;
    int opt = INT_MIN;

    while(i < len || j < len) {
      if (i >= len) {
        break;
      } else if (j >= len) {
        open++;
        if (open > max_open) {
          max_open = open;
          opt = sorted_min[i];
        }
        i++;
      } else {
        if (sorted_min[i] <= sorted_max[j]) {
          open++;
          if (open > max_open) {
            max_open = open;
            opt = sorted_min[i];
          }
          i++;
        } else {
          open--;
          j++;
        }
      }
    }

    support[e] = 0;
    avg_dist[e] = 0;
    min_dist[e] = INT_MIN;
    max_dist[e] = INT_MAX;

    it1 = mate_min[e].begin(); it2 = mate_max[e].begin();
    it3 = mate_avg[e].begin();
    while(it1 != mate_min[e].end() && it2 != mate_max[e].end() &&
          it3 != mate_avg[e].end()) {
      if (*it1 <= opt && *it2 >= opt) {
        support[e]++;
        avg_dist[e] += *it3;
        if (*it1 > min_dist[e])
          min_dist[e] = *it1;
        if (*it2 < max_dist[e])
          max_dist[e] = *it2;
      }
      it1++;
      it2++;
      it3++;
    }
    if (support[e] != max_open)
      printf("%d %d\n", support[e], max_open);
  }

  delete [] sorted_min;
  delete [] sorted_max;
#endif

  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    avg_dist[e] = avg_dist[e] / support[e];
  }

  printf("Added %d edges\n", ecount);

#ifdef DEBUG
  for(lemon::ListGraph::EdgeIt e(G); e != lemon::INVALID; ++e) {
    printf("(%d:%d, %d:%d) support: %d, span min: %d, %d span max: %d, %d dist min: %d, dist max: %d, dist avg: %d\n", 
           node2id[G.u(e)]/2+1, node2id[G.u(e)]%2, node2id[G.v(e)]/2+1, node2id[G.v(e)]%2, support[e], 
           min_span1[e], min_span2[e], max_span1[e], max_span2[e], 
           min_dist[e], max_dist[e], avg_dist);
  }
#endif

}

lemon::ListGraph::NodeMap<int> node2ilp(G);
lemon::ListGraph::EdgeMap<int> edge2ilp(G);

void solve_ilp(ofstream *solution, long genome_length, int *contig_length,
               long component_label, long *labels, list<long> *bnodes, int id) {
  int num_nodes = 0;
  int num_edges = 0;

  int id1, id2, swap;
  int i,j;

  lprec *lp;
  char buf[1024];

  int *colno = NULL;
  REAL *row = NULL;

  long C = 2*genome_length;

  for(list<long>::iterator it = bnodes->begin(); it != bnodes->end(); it++) {
    num_nodes++;
    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    
        num_edges++;
      }
    }

    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)+1]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    
        num_edges++;
      }
    }
  }

#ifdef LP_RELAX
  lp =make_lp(0, num_nodes + num_edges);
  colno = (int *)malloc((num_nodes + num_edges)*sizeof(int));
  row = (REAL *)malloc((num_nodes + num_edges)*sizeof(REAL));
#else
  lp =make_lp(0, 2*num_nodes + num_edges);
  colno = (int *)malloc((2*num_nodes + num_edges)*sizeof(int));
  row = (REAL *)malloc((2*num_nodes + num_edges)*sizeof(REAL));
#endif

  if (lp == NULL || colno == NULL || row == NULL) {
    printf("Could not malloc space for lp model\n");
    abort();
  }

  // Variable names and ranges
  i = 0;
  j = 0;
  for(list<long>::iterator it = bnodes->begin(); it != bnodes->end(); it++) {
    i++;
    sprintf(buf, "x%d", (int)*it);
#ifdef LP_RELAX
    set_col_name(lp, i, buf);
    node2ilp[nodes[2*(*it)]] = i;
    node2ilp[nodes[2*(*it)+1]] = i;
    if (i == 1) {
      set_bounds(lp, i, genome_length/2, genome_length);
    } else {
      set_bounds(lp, i, -genome_length, genome_length);
    }
#else
    set_col_name(lp, 2*i, buf);
    sprintf(buf, "o%d", (int)*it);
    set_col_name(lp, 2*i-1, buf);
    node2ilp[nodes[2*(*it)]] = 2*i;
    node2ilp[nodes[2*(*it)+1]] = 2*i;
    set_bounds(lp, 2*i, 0, genome_length);
    set_binary(lp, 2*i-1, TRUE);
#endif

    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (id1 > id2) {
        swap = id1;
        id1 = id2;
        id2 = swap;
      }

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    
        j++;
        sprintf(buf, "I%d_%d", id1, id2);
#ifdef LP_RELAX
        set_col_name(lp, num_nodes + j, buf);
        edge2ilp[e] = num_nodes + j;
        set_bounds(lp, num_nodes+j, 0, 1);
#else
        set_col_name(lp, 2*num_nodes + j, buf);
        edge2ilp[e] = 2*num_nodes + j;
        set_bounds(lp, 2*num_nodes+j, 0, 1);
        // set_binary(lp, 2*num_nodes+j, TRUE);
#endif
      }
    }

    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)+1]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (id1 > id2) {
        swap = id1;
        id1 = id2;
        id2 = swap;
      }

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    
        j++;
        sprintf(buf, "I%d_%d", id1, id2);
#ifdef LP_RELAX
        set_col_name(lp, num_nodes + j, buf);
        edge2ilp[e] = num_nodes + j;
        set_bounds(lp, num_nodes+j, 0, 1);
#else
        set_col_name(lp, 2*num_nodes + j, buf);
        edge2ilp[e] = 2*num_nodes + j;
        set_bounds(lp, 2*num_nodes+j, 0, 1);
        // set_binary(lp, 2*num_nodes+j, TRUE);
#endif
      }
    }
  }

  i = 0;

  // Objective function
  for(list<long>::iterator it = bnodes->begin(); it != bnodes->end(); it++) {
    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    
        colno[i] = edge2ilp[e];
        row[i] = support[e];
        i++;
      }
    }

    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)+1]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {
        colno[i] = edge2ilp[e];
        row[i] = support[e];
        i++;
      }
    }
  }

  if (!set_obj_fnex(lp, i, row, colno)) {
    printf("LP model building failed\n");
    abort();
  }

  set_maxim(lp);

  // Constraints
  set_add_rowmode(lp, TRUE);

  for(list<long>::iterator it = bnodes->begin(); it != bnodes->end(); it++) {
    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (id2 < id1) {
        swap = id1;
        id1 = id2;
        id2 = swap;
      }

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    

        /* --> --> */
        if ((id1 & 0x01) && !(id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d --> --> %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* x1 + l1 + c1_2 - C + C I1_2 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - (min_dist[e] + max_dist[e])/2);
#endif
          
          /* x1 + l1 + d1_2 + C - C I1_2 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, - C - contig_length[id1/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, - C - contig_length[id1/2] - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1-o2-1+I1_2 <= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 1);

          /* o1-o2+1-I_1_2 >= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, -1);

          /* x1 + l1 + c1_2 - C + C I1_2 - C + C o1 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - avg_dist[e] + 2*C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 + 2*C);
#endif

          /* x1 + l1 + d1_2 + C - C I1_2 + C - C o1 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - avg_dist[e] - 2*C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 - 2*C);
#endif

          /* x2 + l1 + c1_2 - C + C I1_2 - C o1 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x2 + l1 + d1_2 + C - C I1_2 + C o1 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

#endif
        }


        /* --> <-- */
        if ((id1 & 0x01) && (id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d --> <-- %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* x1 + l1 + c1_2 + l2 - C + C I1_2 <= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

          /* x1 + l1 + d1_2 + l2 + C - C I1_2 >= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id1/2] - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id1/2] - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1+o2-1+I1_2 <= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 2);

          /* o1+o2+1-I_1_2 >= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, 0);

          /* x1 + l1 + c1_2 + l2 - C + C I1_2 - C o2 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - contig_length[id2/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x1 + l1 + d1_2 +l2 + C - C I1_2 + C o2 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - contig_length[id2/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

          /* x2 + l2 + c1_2 +l1 - C + C I1_2 - C o1 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - contig_length[id1/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x2 + l2 + d1_2 +l1 + C - C I1_2 + C o1 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - contig_length[id1/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

#endif
        }

        /* <-- --> */
        if (!(id1 & 0x01) && !(id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d <-- --> %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* -x1 + c1_2 - C + C I1_2 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - (min_dist[e] + max_dist[e])/2);
#endif

          /* -x1 + d1_2 + C - C I1_2 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, -C - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, -C - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1+o2-1+I1_2 <= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 2);

          /* o1+o2+1-I_1_2 >= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, 0);

          /* x1 + c1_2 - C + C I1_2 - C o1 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x1 + d1_2 + C - C I1_2 + C o1 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, - (min_dist[e]+max_dist[e])/2 - C);
#endif

          /* x2 + c1_2 - C + C I1_2 - C o2 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, - (min_dist[e]+max_dist[e])/2 + C);
#endif


          /* x2 + d1_2 + C - C I1_2 + C o2 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, - (min_dist[e]+max_dist[e])/2 - C);
#endif

#endif
        }

        /* <-- <-- */
        if (!(id1 & 0x01) && (id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d <-- <-- %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* -x1 + c1_2 + l2 - C + C I1_2 <= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

          /* -x1 + d1_2 + l2 + C - C I1_2 >= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1-o2-1+I1_2 <= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 1);

          /* o1-o2+1-I_1_2 >= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, -1);

          /* x1 + l2 + c1_2 - C + C I1_2 - C o1 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x1 + l2 + d1_2 + C - C I1_2 + C o1 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

          /* x2 + l2 + c1_2 - C + C I1_2 - C + C o1 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - avg_dist[e] + 2*C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 + 2*C);
#endif

          /* x2 + l2 + d1_2 + C - C I1_2 + C  - C o2 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - avg_dist[e]- 2*C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 - 2*C);
#endif

#endif
        }
      }
    }

    for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*(*it)+1]); e != lemon::INVALID; ++e) {
      id1 = node2id[G.u(e)];
      id2 = node2id[G.v(e)];

      // Every edge is traversed twice, skip one traversal
      if (id2/2 == *it)
        continue;

      if (id2 < id1) {
        swap = id1;
        id1 = id2;
        id2 = swap;
      }

      if (labels[id1/2] == component_label && 
          labels[id2/2] == component_label) {    

        /* --> --> */
        if ((id1 & 0x01) && !(id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d --> 2 --> %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* x1 + l1 + c1_2 - C + C I1_2 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - avg_dist[e]);
#else
           add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - (min_dist[e] + max_dist[e])/2);
#endif
          
          /* x1 + l1 + d1_2 + C - C I1_2 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, - C - contig_length[id1/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, - C - contig_length[id1/2] - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1-o2-1+I1_2 <= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 1);

          /* o1-o2+1-I_1_2 >= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, -1);

          /* x1 + l1 + c1_2 - C + C I1_2 - C + C o1 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - avg_dist[e] + 2*C);
#else
           add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 + 2*C);
#endif

          /* x1 + l1 + d1_2 + C - C I1_2 + C - C o1 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - avg_dist[e] - 2*C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 - 2*C);
#endif

          /* x2 + l1 + c1_2 - C + C I1_2 - C o1 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x2 + l1 + d1_2 + C - C I1_2 + C o1 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

#endif
        }


        /* --> <-- */
        if ((id1 & 0x01) && (id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d --> 2 <-- %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* x1 + l1 + c1_2 + l2 - C + C I1_2 <= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id1/2] - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

          /* x1 + l1 + d1_2 + l2 + C - C I1_2 >= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id1/2] - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id1/2] - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1+o2-1+I1_2 <= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 2);

          /* o1+o2+1-I_1_2 >= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, 0);

          /* x1 + l1 + c1_2 +l2 - C + C I1_2 - C o2 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - contig_length[id2/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id1/2] - contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x1 + l1 + d1_2 +l2 + C - C I1_2 + C o2 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - contig_length[id2/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id1/2] - contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

          /* x2 + l2 + c1_2 +l1 - C + C I1_2 - C o1 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - contig_length[id1/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x2 + l2 + d1_2 + l1 + C - C I1_2 + C o1 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - contig_length[id1/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - contig_length[id1/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

#endif
        }

        /* <-- --> */
        if (!(id1 & 0x01) && !(id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d <-- 2 --> %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* -x1 + c1_2 - C + C I1_2 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - (min_dist[e] + max_dist[e])/2);
#endif

          /* -x1 + d1_2 + C - C I1_2 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = -1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, -C - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, -C - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1+o2-1+I1_2 <= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 2);

          /* o1+o2+1-I_1_2 >= 1*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = 1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, 0);

          /* x1 + c1_2 - C + C I1_2 - C o1 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x1 + d1_2 + C - C I1_2 + C o1 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, - (min_dist[e]+max_dist[e])/2 - C);
#endif

          /* x2 + c1_2 - C + C I1_2 - C o2 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, - (min_dist[e]+max_dist[e])/2 + C);
#endif


          /* x2 + d1_2 + C - C I1_2 + C o2 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, - (min_dist[e]+max_dist[e])/2 - C);
#endif

#endif
        }

        /* <-- <-- */
        if (!(id1 & 0x01) && (id2 & 0x01)) {
#ifdef DEBUG_LP
          printf("%d <-- 2 <-- %d   %d-%d-%d\n", id1/2, id2/2, min_dist[e], avg_dist[e], max_dist[e]);
#endif
#ifdef LP_RELAX
          /* -x1 + c1_2 + l2 - C + C I1_2 <= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, LE, C - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

          /* -x1 + d1_2 + l2 + C - C I1_2 >= -x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = -1;
          colno[1] = edge2ilp[e];
          row[1] = -C;
          colno[2] = node2ilp[nodes[id2]];
          row[2] = 1;

#ifdef AVG_DIST
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id2/2] - avg_dist[e]);
#else
          add_constraintex(lp, 3, row, colno, GE, -C - contig_length[id2/2] - (min_dist[e] + max_dist[e])/2);
#endif

#else
          /* o1-o2-1+I1_2 <= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = 1;
          
          add_constraintex(lp, 3, row, colno, LE, 1);

          /* o1-o2+1-I_1_2 >= 0*/
          colno[0] = node2ilp[nodes[id1]]-1;
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]]-1;
          row[1] = -1;
          colno[2] = edge2ilp[e];
          row[2] = -1;
          
          add_constraintex(lp, 3, row, colno, GE, -1);

          /* x1 + l2 + c1_2 - C + C I1_2 - C o1 <= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - avg_dist[e] + C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 + C);
#endif

          /* x1 + l2 + d1_2 + C - C I1_2 + C o1 >= x2 */
          colno[0] = node2ilp[nodes[id1]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id2]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - avg_dist[e] - C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 - C);
#endif

          /* x2 + l2 + c1_2 - C + C I1_2 - C + C o1 <= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id1]]-1;
          row[2] = C;
          colno[3] = edge2ilp[e];
          row[3] = C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - avg_dist[e] + 2*C);
#else
          add_constraintex(lp, 4, row, colno, LE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 + 2*C);
#endif

          /* x2 + l2 + d1_2 + C - C I1_2 + C  - C o2 >= x1 */
          colno[0] = node2ilp[nodes[id2]];
          row[0] = 1;
          colno[1] = node2ilp[nodes[id1]];
          row[1] = -1;
          colno[2] = node2ilp[nodes[id2]]-1;
          row[2] = -C;
          colno[3] = edge2ilp[e];
          row[3] = -C;

#ifdef AVG_DIST
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - avg_dist[e] - 2*C);
#else
          add_constraintex(lp, 4, row, colno, GE, -contig_length[id2/2] - (min_dist[e]+max_dist[e])/2 - 2*C);
#endif

#endif
        }
      }
    }
  }

  set_add_rowmode(lp, FALSE);

  // if (id == 731)
  //   write_LP(lp, stdout);

#ifdef DEBUG_LP
  write_LP(lp, stdout);
#endif

  set_verbose(lp, IMPORTANT);

  solve(lp);

  // if (id == 731) {
  //   /* objective value */
  //   printf("Objective value: %f\n", get_objective(lp));
    
  //   /* variable values */
  //   get_variables(lp, row);
  //   for(j = 0; j < 2*num_nodes + num_edges; j++)
  //     printf("%s: %f\n", get_col_name(lp, j + 1), row[j]);
  // }

#ifdef DEBUG_LP
  /* objective value */
  printf("Objective value: %f\n", get_objective(lp));

  /* variable values */
  get_variables(lp, row);
  for(j = 0; j < 2*num_nodes + num_edges; j++)
    printf("%s: %f\n", get_col_name(lp, j + 1), row[j]);
#endif

  /* variable values */
  get_variables(lp, row);
#ifdef LP_RELAX
  for(j = 0; j < num_nodes + num_edges; j++)
    (*solution) << get_col_name(lp, j + 1) << "\t" << row[j] << "\n";
#else
  // for(j = 0; j < 2*num_nodes + num_edges; j++)
  //   (*solution) << get_col_name(lp, j + 1) << "\t" << row[j] << "\n";
  for(j = 0; j < num_nodes; j++) {
    if (row[2*(j+1)-2] >= 0.9) {
      (*solution) << get_col_name(lp, 2*(j + 1)) << "\t" << (row[2*(j+1)-1] + genome_length/2) << "\n";
    } else {
      (*solution) << get_col_name(lp, 2*(j + 1)) << "\t" << -(row[2*(j+1)-1] + genome_length/2) << "\n";
    }
  }

  for(j = 0; j < num_edges; j++) {
    (*solution) << get_col_name(lp, 2*num_nodes + j + 1) << "\t" << row[2*num_nodes+j] << "\n";
  }

#endif

  free(colno);
  free(row);
  delete_lp(lp);
}

struct id_size {
  long id;
  long size;
};

int cmp_size(const void *p1, const void *p2) {
  struct id_size *i1 = (struct id_size *) p1;
  struct id_size *i2 = (struct id_size *) p2;

  return i2->size - i1->size;
}

void solve_lps(char *solution_file, map<long,list<long> * > *blocks,
               long genome_length, int *contig_length, int contig_count) {
  ofstream solution;
  long *labels = new long[contig_count];
  struct id_size *sorted = new struct id_size[blocks->size()];
  int i;
  int count;

  for(i = 0; i < contig_count; i++)
    labels[i] = 0;

  i = 0;
  for(map<long, list<long> *>::iterator it = blocks->begin(); it != blocks->end(); it++) {
    sorted[i].id = it->first;
    sorted[i].size = it->second->size();
    i++;
  }

  count = i;
  qsort(sorted, count, sizeof(struct id_size), cmp_size);

  solution.open(solution_file);

  for(i = 0; i < count; i++) {
    list<long> *l = (*blocks)[sorted[i].id];
    for(list<long>::iterator lit = l->begin(); lit != l->end(); lit++) {
      labels[*lit] = sorted[i].id;
    }
    if (sorted[i].size >= 3) {
      solve_ilp(&solution, genome_length, contig_length, sorted[i].id, labels, l, i+1);
      solution << "*****" << (i+1) << "\n";
    } else {
      lemon::ListGraph::Edge component_edge = lemon::INVALID;
      for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*l->front()]); 
          e != lemon::INVALID; ++e) {
        if (labels[node2contig[G.u(e)]] == sorted[i].id && 
            labels[node2contig[G.v(e)]] == sorted[i].id) {
          component_edge = e;
          break;
        }
      }
      if (component_edge == lemon::INVALID) {
        for(lemon::ListGraph::IncEdgeIt e(G, nodes[2*l->front()+1]); 
            e != lemon::INVALID; ++e) {
          if (labels[node2contig[G.u(e)]] == sorted[i].id && 
              labels[node2contig[G.v(e)]] == sorted[i].id) {
            component_edge = e;
            break;
          }
        }
      }
      if (component_edge == lemon::INVALID) {
        printf("Could not find component edge!\n");
      } else {
        int id1, id2;
        id1 = node2id[G.u(component_edge)];
        id2 = node2id[G.v(component_edge)];

        if (id1 > id2){
          id2 = id1;
          id1 = node2id[G.v(component_edge)];
        }

        /* --> --> */
        if ((id1 & 0x01) && !(id2 & 0x01)) {
          solution << "x" << (id1/2) << "\t" << (genome_length/2) << "\n";
          solution << "x" << (id2/2) << "\t" << 
#ifdef AVG_DIST
            ((genome_length/2) + contig_length[id1/2] + avg_dist[component_edge])
#else
            ((genome_length/2) + contig_length[id1/2] + (min_dist[component_edge] + max_dist[component_edge])/2)
#endif
                << "\n";
        }

        /* --> <-- */
        if ((id1 & 0x01) && (id2 & 0x01)) {
          solution << "x" << (id1/2) << "\t" << (genome_length/2) << "\n";
          solution << "x" << (id2/2) << "\t" << 
#ifdef AVG_DIST
            -((genome_length/2) + contig_length[id1/2] + avg_dist[component_edge] + 
#else
            -((genome_length/2) + contig_length[id1/2] + (min_dist[component_edge] + max_dist[component_edge])/2 + 
#endif
              contig_length[id2/2])
                   << "\n";
        }

        /* <-- --> */
        if (!(id1 & 0x01) && !(id2 & 0x01)) {
          solution << "x" << (id1/2) << "\t" << -(genome_length/2) << "\n";
          solution << "x" << (id2/2) << "\t" << 
#ifdef AVG_DIST
            ((genome_length/2) + avg_dist[component_edge])
#else
            ((genome_length/2) + (min_dist[component_edge] + max_dist[component_edge])/2)
#endif
                << "\n";
        }

        /* <-- <-- */
        if (!(id1 & 0x01) && (id2 & 0x01)) {
          solution << "x" << (id1/2) << "\t" << -(genome_length/2) << "\n";
          solution << "x" << (id2/2) << "\t" << 
#ifdef AVG_DIST
            -((genome_length/2) + contig_length[id2/2] + avg_dist[component_edge])
#else
            -((genome_length/2) + contig_length[id2/2] + (min_dist[component_edge] + max_dist[component_edge])/2)
#endif
                << "\n";
        }

        solution << "I" << id1 << "_" << id2 << "\t1\n";
        solution << "*****" << (i+1) << "\n";
      }
    }
  }

  solution.close();

}
