/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <stdlib.h>
#include <string.h>

#include <map>

#include "graph.h"
#include "io.h"

void usage(char *prog) {
  printf("Usage: %s [options] <library configuration> <contigs> <contig depths> <output>\n\n", prog);

  printf("Options:\n");
  printf("-n <int>      Upper bound for genome length.\n");
  printf("-s <int>      Minimum mate pair support for contig links.\n");
  printf("-m <int>      Maximum size of biconnected components.\n");
  printf("-d <float>    Maximum depth of non-repeat contigs.\n");
  printf("-l <int>      Minimum length of considered contigs.\n");
  printf("-D <int>      Maximum degree for a contig end.\n");
  printf("-neg          Remove all edges suggesting negative distances.\n");
  printf("-c <string>   Contig graph.\n");
  printf("-e <float>    Maximum error allowed when aligning contigs\n");
}

#define MIN_SUPPORT 2

#define MAX_BLOCK_SIZE 50

#define DEPTH_LIMIT -1.0
#define LENGTH_LIMIT 0
#define DEGREE_LIMIT 50

#define DEPTH_MULTIPLIER 2.5

int main(int argc, char *argv[]) {
  int *d;
  int max, i;
  int *contig_length;
  int contig_count;
  char **contig;
  char **contig_name;
  map<string, int> name2id;

  char *mapping_file[MAX_LIBRARIES];
  int insert_size[MAX_LIBRARIES];
  int min_insert[MAX_LIBRARIES];
  int max_insert[MAX_LIBRARIES];
  int lib_count;

  long genome_len = 100000000;
  int min_support = MIN_SUPPORT;
  int max_block_size = MAX_BLOCK_SIZE;
  double depth_limit = DEPTH_LIMIT;
  int length_limit = LENGTH_LIMIT;
  int degree_limit = DEGREE_LIMIT;

  int check_negative_edges = 1;
  double max_align_err = -1.0;

  char *prog = argv[0];
  char *config_file;
  char *contig_file;
  char *depth_file;
  char *scaffold_file;

  while(argc > 4) {
    if (!strcmp(argv[0], "-n")) {
      genome_len = atol(argv[1]);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-s")) {
      min_support = atoi(argv[1]);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-m")) {
      max_block_size = atoi(argv[1]);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-d")) {
      depth_limit = atof(argv[1]);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-l")) {
      length_limit = atoi(argv[1]);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-D")) {
      degree_limit = atoi(argv[1]);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-neg")) {
      check_negative_edges = 0;
    } else if (!strcmp(argv[0], "-e")) {
      max_align_err = atof(argv[1]);
      argv++;
      argc--;
    }
    argv++;
    argc--;
  }

  if (argc != 4) {
    usage(prog);
    abort();
  }

  config_file = argv[0];
  contig_file = argv[1];
  depth_file = argv[2];
  scaffold_file = argv[3];

  contig_count =  read_contigs(contig_file, &contig_length, &contig, 
                               &contig_name,&name2id);

  printf("Read %d contigs\n", contig_count);

#ifdef DEBUG
  printf("Contig lengths:\n");
  for(i = 0; i < contig_count; i++) {
    printf("%d\t%d\n", i, contig_length[i]);
  }
#endif

  lib_count = read_config_file(config_file, mapping_file, 
                               insert_size, min_insert, max_insert);

  printf("Found %d libraries\n", lib_count);

  printf("Constructing scaffold graph\n");

  for(int i = 0; i < contig_count; i++) {
    nodes[2*i] = G.addNode();
    node2id[nodes[2*i]] = 2*i;
    node2contig[nodes[2*i]] = i;
    nodes[2*i+1] = G.addNode();
    node2id[nodes[2*i+1]] = 2*i+1;
    node2contig[nodes[2*i+1]] = i;
  }

  construct_scaffold_graph(mapping_file, contig_count, contig_length, 
                           &name2id, insert_size, min_insert, 
                           max_insert, lib_count);


  // for(int i = 0; i < lib_count; i++) {
  //   printf("Reading mappings %s\n", mapping_file[i]);
  //   construct_scaffold_graph(mapping_file[i], contig_count, contig_length, 
  //                            &name2id, insert_size[i], min_insert[i], 
  //                            max_insert[i]);
  // }

  max = degree_histogram(&d);

  printf("Degree histogram:\n");
  for(i = 0; i <= max; i++) {
    if (d[i] > 0)
      printf("%d\t%d\n", i, d[i]);
  }

  //  fix_impossible_edges();

  remove_negative_edges(contig, contig_length, min_support, check_negative_edges, max_align_err);

  disconnect_high_degree_nodes(degree_limit);

  double *contig_depth = new double[contig_count];

  read_contig_depth_info(depth_file, &name2id, contig_name, contig_depth);

  if (depth_limit == DEPTH_LIMIT) {
    double sum = 0.0;
    for(i = 0; i < contig_count; i++)
      sum += contig_depth[i];
    sum = sum / (double)contig_count;
    depth_limit = DEPTH_MULTIPLIER*sum;
  }

  printf("Using coverage limit %f\n", depth_limit);

  printf("Removing low support links\n");
  map<long,list<long> * > *blocks;
  blocks = remove_low_support_links(contig_depth, contig_length, min_support,
                                    max_block_size, depth_limit, length_limit);

  max = degree_histogram(&d);

  printf("Degree histogram:\n");
  for(i = 0; i <= max; i++) {
    if (d[i] > 0)
      printf("%d\t%d\n", i, d[i]);
  }

  int *component_size = new int[contig_count];
  int num_components = 0;

  max = 0;

  for(map<long, list<long> *>::iterator it = blocks->begin(); it != blocks->end(); it++) {
    if ((int)it->second->size() > max)
      max = it->second->size();
    component_size[num_components] = it->second->size();
    num_components++;
  }

  d = new int[max+1];
  if (d == NULL)
    abort();

  for(i = 0; i <= max; i++)
    d[i] = 0;

  for(i = 0; i < num_components; i++) {
    d[component_size[i]]++;
  }

  printf("%d components\n", num_components);
  printf("Histogram of component sizes:\n");

  for(i = 0; i <= max; i++) {
    if (d[i] > 0)
      printf("%d\t%d\n", i, d[i]);
  }

  solve_lps(scaffold_file, blocks, genome_len, contig_length, contig_count);

  return 0;
}
