/**
 * Copyright (c) 2011, Leena Salmela <leena.salmela@cs.helsinki.fi>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <iostream>
#include <fstream>

#include <string.h>
#include <stdlib.h>

#include "io.h"
#include "align.h"

#define BUF_LEN 1024

#define MAX_CONTIG_LEN 5000000
#define MAX_SCAFFOLD_LEN 100000000

void patch_scaffold(char *scaffold, int end, char *contig, int ol) {
  int i;

  for(i = ol-1; i >= 0; i--) {
    if (scaffold[end-ol+i] != 'N') {
      return;
    }
  }

  for(i = ol-1; i >= 0; i--) {
    scaffold[end-ol+i] = contig[i];
  }

  // int i, j;
  // int n = 0;

  // for(i = ol; i >= 0; i--) {
  //   if (scaffold[end-ol+i] == 'N') {
  //     n++;
  //   } else {
  //     n = 0;
  //   }

  //   if (n > 10) {
  //     for(j = i; j >= 0; j--) {
  //       if (scaffold[end-ol+j] != 'N')
  //         break;
  //     }
  //     j += 11;

  //     for(; j <= i; j++) {
  //       scaffold[end-ol+j] = contig[j];
  //     }

  //     i = j;
  //     n = 0;
  //   }

  // }

}

void translate_scaffolds(char *sc_input, char *sc_output, char *pl_output,
                         int *contig_length, 
                         char **contig_seq, char **contig_names, 
                         map<string,int> *contig2id, double max_align_err) {

  ifstream input;
  ofstream output;
  ofstream output2;
  char buffer[BUF_LEN];
  char *scaffold = new char[MAX_SCAFFOLD_LEN];

  char *reverse = new char[MAX_CONTIG_LEN];

  long offset = -1;
  long padding = 0;
  int ok = 0;
  int sc_id;
  int num_contigs;
  long end = 0;
  int i;
  align_t *align = NULL;

  align = init_align();

  input.open(sc_input);
  output.open(sc_output);
  output2.open(pl_output);

  while (!(input).getline(buffer, BUF_LEN).eof()) {
    if (memcmp(buffer, "Scaffold ", 9) == 0) {
      if (ok) {
        scaffold[end] = '\0';
        output << scaffold;
        output << "\n";
      }

      end = 0;
      padding = 0;
      offset = -1;
      sc_id = atoi(&buffer[9]);
      for(i = 0; i < BUF_LEN; i++)
        if (buffer[i] == '=')
          break;
      num_contigs = atoi(&buffer[i+2]);
      
      if (num_contigs >= 1) {
        ok = 1;
        offset = -1;
        end = 0;
        padding = 0;
        output << ">scaffold" << sc_id << "_contigs=" << num_contigs << "\n";
        output2 << "scaffold" << sc_id << "_contigs=" << num_contigs << "\n";
      } else {
        ok = 0;
      }
    } else if (ok && (memcmp(buffer, "  ", 2) == 0)) {
      long s, e, ol;
      int id;
      for(i = 0; i < BUF_LEN; i++)
        if (buffer[i] == ':') {
          buffer[i] = '\0';
          break;
        }
      id = atoi(&buffer[2]);
      
      s = atol(&buffer[i+2]);
      for(i = i+2; i < BUF_LEN; i++)
        if (buffer[i] == '\t')
          break;
      e = atol(&buffer[i+1]);

      s += padding;
      e += padding;

      if (offset == -1) {
        if (s < e)
          offset = s;
        else
          offset = e;
      }

      if (s < e) {
        // check for overlap
        if (s - offset < end) {
          if (e-offset > end) {
            ol = is_overlap(scaffold, 1, end, contig_seq[id],
                            1, contig_length[id], 
                            s-offset-500, s-offset+500, align, max_align_err);

            if (ol) {
              output2 << "contig\t" << contig_names[id] << "\tF\t" << 
                (end-ol) << "\t" << (end-ol+contig_length[id]) << "\n";
              // patch_scaffold(scaffold, end, contig_seq[id], ol);
              strcpy(&scaffold[end], &contig_seq[id][ol]);
              padding = (end-ol) - (s-offset);
              end += (contig_length[id] -ol);
            } else {
              // No overlap -> print gap
              for(i = 0; i < 10; i++) {
                scaffold[end++] = 'N';
              }
              // print contig
              output2 << "contig\t" << contig_names[id] << "\tF\t" << 
                end <<  "\t" << (end+contig_length[id]) << "\n";
              strcpy(&scaffold[end], contig_seq[id]);
              padding = end - (s-offset);
              end += contig_length[id];
            }
          } else {
            output2 << "contig\t" << contig_names[id] << "\tF\t" << 
              (s-offset) <<  "\t" << (e-offset) << "\n";
            patch_scaffold(scaffold, e-offset, contig_seq[id], e-s);
          }
        } else {
          // print gap
          for(i = end; i < s-offset; i++) {
            scaffold[end++] = 'N';
          }
          // print contig
          output2 << "contig\t" << contig_names[id] << "\tF\t" << 
            end <<  "\t" << (end+contig_length[id]) << "\n";
          strcpy(&scaffold[end], contig_seq[id]);
          end += contig_length[id];
        }
        
      } else {
        // check for overlap
        if (e - offset < end) {
          if (s - offset > end) {
            ol = is_overlap(scaffold, 1, end, contig_seq[id],
                            0, contig_length[id],
                            e-offset-500, e-offset+500, align, max_align_err);
            if (ol) {
              output2 << "contig\t" << contig_names[id] << "\tR\t" << 
                (end-ol) <<  "\t" << (end-ol+contig_length[id]) << "\n";
              reverse_base(contig_seq[id], reverse);
              // patch_scaffold(scaffold, end, reverse, ol);
              strcpy(&scaffold[end], &reverse[ol]);
              padding = (end-ol) - (e-offset);
              end += (contig_length[id] - ol);
            } else {
              // No overlap -> print gap
              for(i = 0; i < 10; i++) {
                scaffold[end++] = 'N';
              }
              // print contig
              output2 << "contig\t" << contig_names[id] << "\tR\t" << 
                end <<  "\t" << (end+contig_length[id]) << "\n";
              reverse_base(contig_seq[id], &scaffold[end]);
              padding = end - (e-offset);
              end += contig_length[id];
            }
          } else {
            output2 << "contig\t" << contig_names[id] << "\tR\t" << 
              (e-offset) <<  "\t" << (s-offset) << "\n";
            reverse_base(contig_seq[id], reverse);
            patch_scaffold(scaffold, s-offset, reverse, s-e);
          }
        } else {
          // print gap
          for(i = end; i < e-offset; i++) {
            scaffold[end++] = 'N';
          }
          // print contig
          output2 << "contig\t" << contig_names[id] << "\tR\t" << 
            end <<  "\t" << (end+contig_length[id]) << "\n";
          reverse_base(contig_seq[id], &scaffold[end]);
          end += contig_length[id];
        }
      }
    }
  }

  if (ok) {
    scaffold[end] = '\0';
    output << scaffold;
    output << "\n";
  }

  input.close();
  output.close();

}
