/*
 * Copyright (C) 2002  Pekka Enberg <Pekka.Enberg@cs.helsinki.fi>
 *
 * Distributed under the terms of the GNU General Public License
 * version 2 or later.
 */
#include <assert.h>
#include <curses.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include "error.h"
#include "colors.h"

void
set_attribute (int attribute)
{
  int fg_color = (attribute & 0x0F);
  int bg_color = (attribute & 0xF0) >> 4;

  /* We need to turn on the BOLD attribute to get colors > 7.  */
  if (fg_color > 7)
    {
      attron (A_BOLD);
      fg_color -= 7;
    }
  else
    {
      attroff (A_BOLD);
    }
  attron (COLOR_PAIR (attr_to_color_pair(fg_color, bg_color)));
}

#define COLOR_ATTR(fg, bg) (bg * 0x10 + fg)

/*
 * If you use any of the colors, please define a constant here instead
 * of hardcoding the value into the code.
 */
#define RED_ON_BLACK 1

void
print_status (int screen_height, size_t x, size_t y,
	      int fg_color, int bg_color)
{
  move (screen_height - 1, 0);
  clrtoeol ();

  attron (COLOR_PAIR(RED_ON_BLACK));
  mvprintw (screen_height - 1, 1, "(%2i, %2i)", x, y);
  attroff (COLOR_PAIR(RED_ON_BLACK));

  move (screen_height - 1, 13);
  set_attribute (COLOR_ATTR (fg_color, bg_color));
  printw ("Color");
}

struct edit_buffer {
  size_t height;
  size_t width;
  int * buffer;
};

void
edit_buffer_put (struct edit_buffer * buffer, size_t x, size_t y, int value)
{
  assert (x < buffer->width ); assert (x >= 0);
  assert (y < buffer->height); assert (y >= 0);

  buffer->buffer[y * buffer->width + x] = value;
}

int
edit_buffer_get (struct edit_buffer * buffer, size_t x, size_t y)
{
  assert (x < buffer->width ); assert(x >= 0);
  assert (y < buffer->height); assert(y >= 0);

  return buffer->buffer[y * buffer->width + x];
}

void
edit_buffer_clear (struct edit_buffer * buffer)
{
  int x, y;

  for (y = 0; y < buffer->height; y++)
    for (x = 0; x < buffer->width; x++)
      edit_buffer_put (buffer, x, y, 0x0720);
}

void
edit_buffer_draw_to_screen (struct edit_buffer * buffer, size_t start_x,
			    size_t start_y, size_t screen_width,
			    size_t screen_height)
{
  assert (start_x + screen_width  <= buffer->width );
  assert (start_y + screen_height <= buffer->height);
  assert (start_x >= 0);
  assert (start_y >= 0);

  int x, y;

  for (y = 0; y < screen_height; y++)
    for (x = 0; x < screen_width; x++)
      {
	int attribute =
	  (edit_buffer_get (buffer, start_x + x, start_y + y) & 0xFF00) >> 8;
	int character =
	  edit_buffer_get (buffer, start_x + x, start_y + y) & 0xFF;

	set_attribute (attribute);
	mvprintw (y, x, "%c", character);
      }
}

void
edit_buffer_init (struct edit_buffer * buffer, size_t width, size_t height)
{
  buffer->buffer = malloc(height * width * sizeof(int));
  if (buffer->buffer == NULL)
    {
      error("Could not allocate memory for edit buffer.");
    }
  buffer->height = height;
  buffer->width = width;
}

void
edit_buffer_release (struct edit_buffer * buffer)
{
  if (buffer->buffer != NULL)
    free(buffer->buffer);

  buffer->buffer = NULL;
  buffer->width = 0;
  buffer->height = 0;
}

/* Curses type KEY_xxx check if META key was used for character.  */
#define KEY_META(ch) (0x1000 | ch)

/* ALT and ESC are both the same META key for terminals.  */
#define META_KEY_CODE 0x1B

int
get_char ()
{
  int ch = getch ();
  if (ch == META_KEY_CODE)
    {
      /* After META comes the actual key we're interested in.  */
      ch = KEY_META(getch());
    }
  return ch;
}

int
get_printable_char (int ch)
{
  /* TODO: Function keys should go trough lookup table.  */
  switch (ch)
    {
    case KEY_F(1):
      return 176;
    case KEY_F(2):
      return 177;
    case KEY_F(3):
      return 178;
    case KEY_F(4):
      return 219;
    case KEY_F(5):
      return 220;
    case KEY_F(6):
      return 223;
    case KEY_F(7):
      return 221;
    case KEY_F(8):
      return 222;
    case KEY_F(10):
      return 254;
    }
  return ch;
}

/*
 * Global variables are _evil_ so don't you dare to touch these
 * outside cmd_* functions.
 */

static bool quit = false;

static int fg_color = 0x07;
static int bg_color = 0x00;

static int edit_buffer_start_x = 0;
static int edit_buffer_start_y = 0;
static int screen_x = 0;
static int screen_y = 0;

static int screen_height;
static int screen_width;

static struct edit_buffer buffer;

/*
 * Commands issued in editor
 */

void
cmd_quit ()
{
  quit = true;
}

void
cmd_resize ()
{
  /* Currently we don't support resizing the terminal while
     running.  */
  error ("Terminal was resized.");
}

void
cmd_move_up ()
{
  if (screen_y > 0)
    screen_y--;
  else if (edit_buffer_start_y > 0)
    edit_buffer_start_y--;
}

void
cmd_move_down ()
{
  if (screen_y < (screen_height - 1))
    screen_y++;
  else if (edit_buffer_start_y + screen_height < buffer.height)
    edit_buffer_start_y++;
}

void
cmd_move_left ()
{
  if (screen_x > 0)
    screen_x--;
  else if (edit_buffer_start_x > 0)
    edit_buffer_start_x--;
}

void
cmd_move_right ()
{
  if (screen_x < (screen_width - 1))
    screen_x++;
  else if (edit_buffer_start_x < (buffer.width - screen_width))
    edit_buffer_start_x++;
}

void
cmd_move_to_end ()
{
  screen_x = (screen_width - 1);
  edit_buffer_start_x = buffer.width - screen_width;
}

void
cmd_move_to_start ()
{
  screen_x = 0;
  edit_buffer_start_x = 0;
}

void
cmd_print_char (int ch)
{
  edit_buffer_put (&buffer,
		   edit_buffer_start_x + screen_x,
		   edit_buffer_start_y + screen_y,
		   (COLOR_ATTR (fg_color, bg_color) << 8) |
		   get_printable_char (ch));
}

void
cmd_move_page_up ()
{
  edit_buffer_start_y -= screen_height;
  if (edit_buffer_start_y < 0)
    {
      edit_buffer_start_y = 0;
      screen_y = 0;
    }
}

void
cmd_move_page_down ()
{
  edit_buffer_start_y += screen_height;
  if (edit_buffer_start_y > buffer.height - screen_height)
    {
      edit_buffer_start_y = buffer.height - screen_height;
      screen_y = (screen_height - 1);
    }
}

#define MAX_BG_COLOR 7
#define MAX_FG_COLOR 14

#define DECREASE_AND_WRAP(c, max) (c >   0 ? c - 1 : max)
#define INCREASE_AND_WRAP(c, max) (c < max ? c + 1 :   0)

void
cmd_next_fg_color ()
{
  fg_color = INCREASE_AND_WRAP (fg_color, MAX_FG_COLOR);
}

void
cmd_prev_fg_color ()
{
  fg_color = DECREASE_AND_WRAP (fg_color, MAX_FG_COLOR);
}

void
cmd_next_bg_color ()
{
  bg_color = INCREASE_AND_WRAP (bg_color, MAX_BG_COLOR);
}

void
cmd_prev_bg_color ()
{
  bg_color = DECREASE_AND_WRAP (bg_color, MAX_BG_COLOR);
}

/*
 * Main editor loop
 */

void
edit_loop ()
{
  getmaxyx (stdscr, screen_height, screen_width);
  if (screen_height == -1 || screen_width == -1)
    {
      error("Could not get screen dimensions from curses.");
    }

  /* Leave a line for the status bar.  */
  screen_height--;

  edit_buffer_init (&buffer, 160, 100);
  edit_buffer_clear (&buffer);

  while (!quit)
    {
      edit_buffer_draw_to_screen (&buffer,
				  edit_buffer_start_x,
				  edit_buffer_start_y,
				  screen_width,
				  screen_height);

      print_status (screen_height + 1,
		    edit_buffer_start_x + screen_x + 1,
		    edit_buffer_start_y + screen_y + 1,
		    fg_color,
		    bg_color);
      
      move (screen_y, screen_x);

      int ch = get_char ();
      if (ch == ERR)
	{
	  error ("getch() returned ERR");
	}
      switch (ch)
	{
	case KEY_META('x'):
	case KEY_META('X'): cmd_quit (); break;

	case KEY_RESIZE: cmd_resize (); break;
	case KEY_UP:     cmd_move_up (); break;
	case KEY_DOWN:   cmd_move_down (); break;
	case KEY_LEFT:   cmd_move_left (); break;
	case KEY_RIGHT:  cmd_move_right (); break;
	case KEY_END:    cmd_move_to_end (); break;
	case KEY_HOME:   cmd_move_to_start (); break;
	case KEY_PPAGE:  cmd_move_page_up (); break;
	case KEY_NPAGE:  cmd_move_page_down (); break;

	case KEY_META(KEY_UP):    cmd_next_bg_color (); break;
	case KEY_META(KEY_DOWN):  cmd_prev_bg_color (); break;
	case KEY_META(KEY_RIGHT): cmd_next_fg_color (); break;
	case KEY_META(KEY_LEFT):  cmd_prev_fg_color (); break;

	case KEY_BACKSPACE:
	  cmd_move_left ();
	  cmd_print_char (' ');
	  break;

	default:
	  cmd_print_char (ch);
	  cmd_move_right ();
	}
    }
  edit_buffer_release (&buffer);
}

void
init_curses ()
{
  savetty ();
  initscr ();
  nonl ();
  cbreak ();
  noecho ();
  keypad (stdscr, TRUE);

  if (has_colors() == FALSE)
    {
      error("Your terminal doesn't support colors.");
    }
  start_color ();
}

void
release_curses ()
{
  resetty ();
  endwin ();
}

int
main ()
{
  init_curses ();
  init_color_pairs ();
  edit_loop();
  release_curses ();
  printf ("Bye, bye.\n");

  return EXIT_SUCCESS;
}

/*
 * Local Variables:
 * c-file-style: "gnu"
 * End:
 */
