/*
 * Copyright (C) 2002-2003  Pekka Enberg <penberg@iki.fi>
 *
 * Distributed under the terms of the GNU General Public License
 * version 2 or later.
 */
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include "buffer.h"
#include "indent.h"

/*
 * Declarations used by lexer
 */
static void xml_declaration(struct buffer * buffer, const char * text);
static void cdata_section(struct buffer * buffer, const char * text);
static void doctype_declaration(struct buffer * buffer, const char * text);
static void start_tag(struct buffer * buffer, const char * text);
static void end_tag(struct buffer * buffer, const char * text);
static void empty_element_tag(struct buffer * buffer, const char * text);
static void comment(struct buffer * buffer, const char * text);
static void newline(struct buffer * buffer, const char * text);
static void do_newline(struct buffer * buffer, const char * text);
static void character_text(struct buffer * buffer, const char * text);

/*
 * Uh, oh, we're communicating with yacc trough a global variable
 * called "buffer".
 */
struct buffer __buffer;
struct buffer * buffer = &__buffer;

#include "lex.yy.c"

static struct indent_options indent_opts;
static FILE * indent_output;

#define DEFAULT_INDENT_CHAR ' '
#define DEFAULT_NUM_INDENT_CHARS 4

/* Set default indent options.  */
void indent_options_set_defaults(struct indent_options * opts)
{
    opts->indent_char      = DEFAULT_INDENT_CHAR;
    opts->num_indent_chars = DEFAULT_NUM_INDENT_CHARS;
    opts->max_columns      = -1;
    opts->wrap_long_lines  = false;
    opts->force_newline_after_start_tag  = false;
    opts->force_newline_after_end_tag    = false;
    opts->force_newline_before_start_tag = false;
    opts->force_newline_before_end_tag   = false;
}

static void set_options(struct indent_options * opts)
{
    indent_opts.indent_char      = opts->indent_char;
    indent_opts.num_indent_chars = opts->num_indent_chars;
    indent_opts.max_columns      = opts->max_columns;
    indent_opts.wrap_long_lines  = opts->wrap_long_lines;
    indent_opts.force_newline_after_start_tag  = opts->force_newline_after_start_tag;
    indent_opts.force_newline_after_end_tag    = opts->force_newline_after_end_tag;
    indent_opts.force_newline_before_start_tag = opts->force_newline_before_start_tag;
    indent_opts.force_newline_before_end_tag   = opts->force_newline_before_end_tag;
}

#define BUFFER_INITIAL_CAPACITY 1024

void indent(FILE * input, FILE * output, struct indent_options * opts)
{
    set_options(opts);
    buffer_init(buffer, BUFFER_INITIAL_CAPACITY);

    yyin = input;
    indent_output = output;
    yylex();

    /*
     * There might not have been a newline before EOF.
     */
    buffer_flush(buffer, indent_output);
    buffer_release(buffer);
}

/* Print indent characters.  */
static void print_indent(FILE * output, int indent_level)
{
    int i;

    for (i = 0; i < (indent_opts.num_indent_chars * indent_level); i++) {
	fputc(indent_opts.indent_char, output);
    }
}

static void xml_declaration(struct buffer * buffer, const char * text)
{
    buffer_push(buffer, text);
}

static void cdata_section(struct buffer * buffer, const char * text)
{
    buffer_push(buffer, text);
}

static void doctype_declaration(struct buffer * buffer, const char * text)
{
    buffer_push(buffer, text);
}

/* XML end of line characters.  */
#define CARRIAGE_RETURN 0x0D
#define LINE_FEED       0x0A
#define NEL             0x85

static inline bool is_newline(int current)
{
    if ((CARRIAGE_RETURN == current)
	|| (LINE_FEED    == current)
	|| (NEL          == current))
	return true;

    return false;
}

static void force_newline_before(struct buffer * buffer)
{
    int current;

    if (buffer_size(buffer) == 0) {
	/*
	 * We just did a newline, no need to force it.
	 */
	return;
    }

    current = buffer_pop(buffer);
    buffer_push_char(buffer, current);
    /*
     * We need to add a newline only if there isn't one already.
     */
    if (!is_newline(current)) {
	newline(buffer, "\n");
    }
}

static void force_newline_after(struct buffer * buffer)
{
    int current = input();
    /*
     * We need to add a newline only if there isn't one already.
     */
    if (!is_newline(current)) {
	newline(buffer, "\n");
    }
    unput(current);
}

static void force_newline_wrap(struct buffer * buffer)
{
    int current = input();
    /*
     * We need to add a newline only if there isn't one already.
     */
    if (!is_newline(current)) {
	/*
	 * If we're wrapping a line, we don't want to eat any leading
	 * whitespace.
	 */
	do_newline(buffer, "\n");
    }
    unput(current);
}

static int indent_level = 0;
static int indent_level_delta = 0;

static void start_tag(struct buffer * buffer, const char * text)
{
    char * tmp;
    /*
     * Save text because force_newline_before can trash it.
     */
    tmp = strdup(text);

    if (indent_opts.force_newline_before_start_tag)
	force_newline_before(buffer);

    buffer_push(buffer, tmp);
    free(tmp);
    indent_level_delta++;

    if (indent_opts.force_newline_after_start_tag)
	force_newline_after(buffer);
}

static void end_tag(struct buffer * buffer, const char * text)
{
    char * tmp;
    /*
     * Save text because force_newline_before can trash it.
     */
    tmp = strdup(text);

    if (indent_opts.force_newline_before_end_tag)
	force_newline_before(buffer);

    buffer_push(buffer, tmp);
    free(tmp);
    indent_level_delta--;

    if (indent_opts.force_newline_after_end_tag)
	force_newline_after(buffer);
}

static void empty_element_tag(struct buffer * buffer, const char * text)
{
    buffer_push(buffer, text);
}

static int input_and_push(struct buffer * buffer)
{
    int ret = input();
    if (ret != EOF) buffer_push_char(buffer, ret);
    return ret;
}

static void comment(struct buffer * buffer, const char * text)
{
    int c;

    buffer_push(buffer, text);
    for (;;) {
	while ((c = input_and_push(buffer)) != '-' &&
	       c != EOF)
	    ;
	if ((c = input_and_push(buffer)) != '-') {
	    continue;
	}
	if ((c = input_and_push(buffer)) == '>') {
	    break;
	}
    }
}

/* Check for whitespace character.  */
static inline bool is_whitespace(int c)
{
    return ((c == ' ') || (c == '\f') || (c == '\t') || (c == '\v'));
}

/* Eat whitespace from stream.  */
static void eat_whitespace()
{
    for (;;) {
	int current = input();
	if (!is_whitespace(current)) {
	    unput(current);
	    break;
	}
    }
}

static void do_newline(struct buffer * buffer, const char * text)
{
    /*
     * Does not eat whitespace after flushing buffer.
     */ 
    if (indent_level_delta < 0) {
	indent_level += indent_level_delta;
    }
    print_indent(indent_output, indent_level);
    buffer_push(buffer, text);
    buffer_flush(buffer, indent_output);
    if (indent_level_delta > 0) {
	indent_level += indent_level_delta;
    }
    indent_level_delta = 0;
}

static void newline(struct buffer * buffer, const char * text)
{
    do_newline(buffer, text);
    eat_whitespace();
}

/*
 * We assume tab is equal to 8 spaces.
 */
#define TAB_SIZE 8

static unsigned long indent_size()
{
    return (indent_opts.indent_char == '\t'
	    ? indent_level * TAB_SIZE
	    : indent_level * indent_opts.num_indent_chars);
}

static bool need_wrap(struct buffer * buffer)
{
    return buffer_size(buffer) + indent_size() == indent_opts.max_columns;
}

static void character_text(struct buffer * buffer, const char * text)
{
    char current;
    int next;

    /*
     * We should get one character at a time.
     */
    assert(strlen(text) == 1);

    /*
     * unput() trashes yytext so lets grab it while we still can
     */
    current = text[0];

    /*
     * Pushing EOF character into the buffer leads to ugly results in
     * the output stream. The only reasonable way to detect EOF
     * condition is to check for the next character.
     */
    next = input();
    if (next == EOF) {
	return;
    }
    unput(next);
    buffer_push_char(buffer, current);

    /*
     * Forcing newline changes 'text' so lets do it after we've pushed
     * it to the buffer.
     */
    if (indent_opts.wrap_long_lines && need_wrap(buffer)) {
	struct buffer tmp;
	buffer_init(&tmp, buffer_size(buffer));
	/*
	 * Find last character that was not whitespace
	 */
	for (;;) {
	    int c;
	    if (buffer_size(buffer) == 0)
		break;

	    c = buffer_pop(buffer);
	    if (is_whitespace(c)) {
		/*
		 * Do not push whitespace because it would appear
		 * after the newline.
		 */
		break;
	    }
	    /*
	     * Characters are put in tmp buffer in reverse order.
	     */
	    buffer_push_char(&tmp, c);
	}
	force_newline_wrap(buffer);
	/*
	 * Restore non-wrapped text into buffer.
	 */
	while (buffer_size(&tmp) > 0) {
	    buffer_push_char(buffer, buffer_pop(&tmp));
	}
	buffer_release(&tmp);
    }
}

