/*
 * Copyright (C) 2002-2003  Pekka Enberg <penberg@iki.fi>
 *
 * Distributed under the terms of the GNU General Public License
 * version 2 or later.
 */
#include "buffer.h"
#include "error.h"
#include "indent.h"
#include "version.h"
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

static void version()
{
    printf("%s %s\n", PACKAGE, VERSION);
}

static void usage(char * argv[])
{
    printf("%s: [-o <output filename>] [-i <indent level>] [-t] "
	   "[-l max columns] [-n <newline pos>] [-h] [-v] [-w]\n",
	   argv[0]);
}

static void parse_force_newline_arg(char * arg, struct indent_options * opts)
{
    if (strcmp(arg, "as") == 0) {
	opts->force_newline_after_start_tag = true;
    } else if (strcmp(arg, "ae") == 0) {
	opts->force_newline_after_end_tag = true;
    } else if (strcmp(arg, "bs") == 0) {
	opts->force_newline_before_start_tag = true;
    } else if (strcmp(arg, "be") == 0) {
	opts->force_newline_before_end_tag = true;
    }
}

int main(int argc, char * argv[])
{
    char input_filename[FILENAME_MAX];
    char output_filename[FILENAME_MAX];
    FILE * input;
    FILE * output;
    int arg_index;
    bool indent_level_explicitly_defined;
    bool overwrite_original_file;
    struct indent_options opts;

    input_filename[0] = '\0';
    output_filename[0] = '\0';
    indent_level_explicitly_defined = false;
    overwrite_original_file = false;
    indent_options_set_defaults(&opts);

    for (;;) {
	arg_index = getopt(argc, argv, "hi:l:o:n:tvw");
	if (arg_index == -1) {
	    break;
	}
	switch (arg_index) {
        case 'h':
	    usage(argv);
	    return EXIT_SUCCESS;
        case 'i':
	    opts.num_indent_chars = atoi(optarg);
	    indent_level_explicitly_defined = true;
	    break;
	case 'l':
	    opts.max_columns = atoi(optarg);
	    opts.wrap_long_lines = true;
	    break;
	case 't':
	    opts.indent_char = '\t';
	    if (!indent_level_explicitly_defined) {
		/* Default indent level for tabs is different.  */
		opts.num_indent_chars = 1;
	    }
	    break;
	case 'o':
	    strcpy(output_filename, optarg);
	    break;
	case 'n':
	    parse_force_newline_arg(optarg, &opts);
	    break;
	case 'v':
	    version();
	    return EXIT_SUCCESS;
	case 'w':
	    overwrite_original_file = true;
	    break;
        default:
	    usage(argv);
	    return EXIT_FAILURE;
	}
    }

    if (argv[optind] != NULL) {
	strcpy(input_filename, argv[optind]);

	if (overwrite_original_file) {
	    char backup_filename[FILENAME_MAX];
	    strcpy(backup_filename, input_filename);
	    strcat(backup_filename, "~");

	    rename(input_filename, backup_filename);
	    strcpy(output_filename, input_filename);
	    strcpy(input_filename, backup_filename);
	}
    }

    if (strlen(input_filename) > 0) {
	input = fopen(input_filename, "r");
	if (NULL == input) {
	    error("Could not open '%s': %s.", input_filename,
		  strerror(errno));
	    return EXIT_FAILURE;
	}
    } else {
	input = stdin;
    }

    if (strlen(output_filename) > 0) {
	output = fopen(output_filename, "w");
	if (NULL == output) {
	    error("Could not open '%s': %s.", output_filename,
		  strerror(errno));
	    return EXIT_FAILURE;
	}
    } else {
	output = stdout;
    }

    indent(input, output, &opts);

    fclose(output);
    fclose(input);

    return EXIT_SUCCESS;
}
